import numpy as np
from pyrep.objects.shape import Shape          # kept from skeleton (may be unused but required)
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import ALL predefined skill functions (move, pick, place, rotate, pull, …)
from skill_code import *

# Video helpers
from video import init_video_writers, recording_step, recording_get_observation

# Utility that provides the 3‑D positions of relevant scene objects
from object_positions import get_object_positions


# ------------------------------------------------------------------------------
# Helper utilities
# ------------------------------------------------------------------------------
def safe_get_pos(position_dict, key):
    """Return the position of `key` or raise a descriptive error."""
    if key not in position_dict:
        raise KeyError(f"[run_task] Could not find position for object '{key}'. "
                       f"Available keys: {list(position_dict.keys())}")
    return position_dict[key]


def run_task():
    """Run the complete oracle plan for the ‘open drawer + move tomatoes’ task."""
    print("===== Starting Combined‑Task Execution =====")

    # ----------------------------------------------------------------------
    # 1)  Environment initialisation
    # ----------------------------------------------------------------------
    env, task = setup_environment()

    try:
        # Reset environment to obtain an initial observation (and description)
        _, obs = task.reset()

        # Video / observation recording (optional – harmless if not supported)
        try:
            init_video_writers(obs)
            task.step = recording_step(task.step)               # wrap the step method
            task.get_observation = recording_get_observation(   # wrap the get_observation method
                task.get_observation
            )
        except Exception as e:
            print("[run_task] Video module not available or failed to initialise – continuing without video.")
            print("            Error:", e)

        # ------------------------------------------------------------------
        # 2)  Retrieve all object positions from helper module
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Mapping between PDDL‑style names (from the specification) and RLBench
        # object names (as provided by object_positions.py & object list).
        name_map = {
            "side-pos-bottom":   "bottom_side_pos",
            "anchor-pos-bottom": "bottom_anchor_pos",
            # tomatoes & plate
            "tomato1": "item1",
            "tomato2": "item2",
            "plate":   "plate",
        }

        # Convenience lambda to fetch mapped positions
        get_pos = lambda pddl_name: safe_get_pos(positions, name_map[pddl_name])

        # ------------------------------------------------------------------
        # 3)  Execute oracle plan step‑by‑step
        # ------------------------------------------------------------------

        # STEP 1  – move to the lower drawer’s side position
        print("\n[Plan] Step 1/8 – move → side‑pos‑bottom")
        obs, reward, done = move(
            env, task,
            target_pos=get_pos("side-pos-bottom")
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 1")
            return

        # STEP 2  – rotate gripper to 90° (about its z‑axis)
        # Build quaternion (xyzw) for 90° rotation around z
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("\n[Plan] Step 2/8 – rotate → 90 deg about z")
        obs, reward, done = rotate(
            env, task,
            target_quat=target_quat
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 2")
            return

        # STEP 3  – move to the anchor position on the same drawer
        print("\n[Plan] Step 3/8 – move → anchor‑pos‑bottom")
        obs, reward, done = move(
            env, task,
            target_pos=get_pos("anchor-pos-bottom")
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 3")
            return

        # STEP 4  – pull the drawer handle outward along −x by 0.25 m
        print("\n[Plan] Step 4/8 – pull drawer 0.25 m along −x")
        obs, reward, done = pull(
            env, task,
            pull_distance=0.25,
            pull_axis='-x'
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 4")
            return

        # STEP 5  – pick up first tomato (item1) from the table
        print("\n[Plan] Step 5/8 – pick tomato1 (item1) from table")
        tomato1_pos = get_pos("tomato1")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 5")
            return

        # STEP 6  – place tomato1 onto the plate
        print("\n[Plan] Step 6/8 – place tomato1 onto plate")
        plate_pos = get_pos("plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 6")
            return

        # STEP 7  – pick up second tomato (item2) from the table
        print("\n[Plan] Step 7/8 – pick tomato2 (item2) from table")
        tomato2_pos = get_pos("tomato2")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done:
            print("[Plan] Task ended unexpectedly at Step 7")
            return

        # STEP 8  – place tomato2 onto the plate
        print("\n[Plan] Step 8/8 – place tomato2 onto plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,             # same plate as before
            approach_distance=0.15,
            approach_axis='z'
        )

        # ------------------------------------------------------------------
        # 4)  Result summary
        # ------------------------------------------------------------------
        print("\n===== Task Finished =====")
        if done:
            print("[run_task] Environment indicated task completion (done=True). "
                  f"Final reward: {reward}")
        else:
            print("[run_task] Plan executed (steps 1‑8) – environment did not signal "
                  "terminal state, but high‑level goal should now be satisfied.")

    # ----------------------------------------------------------------------
    # 5)  Always ensure clean shutdown
    # ----------------------------------------------------------------------
    finally:
        shutdown_environment(env)
        print("===== Environment shut down. =====")


# Driver
if __name__ == "__main__":
    run_task()