# run_skeleton_task.py  (completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *          # <- all predefined primitives (move, pick, place, rotate, pull)

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def _alias_name(name: str) -> str:
    """Translate a symbolic name from the oracle plan to the real shape name
    that exists in the scene."""
    alias_table = {
        # positions for the drawer manipulation
        "side-pos-bottom":   "bottom_side_pos",
        "anchor-pos-bottom": "bottom_anchor_pos",
        "nowhere-pos":       "waypoint1",          # safe parking position
        # objects (plan says ‘tomato’, scene contains ‘item’)
        "tomato1":           "item1",
        "tomato2":           "item2",
        "tomato3":           "item3",
        # plate remains identical
        "plate":             "plate",
    }
    return alias_table.get(name, name)   # default: identity


def _quat_about_z(deg: float) -> np.ndarray:
    """Return quaternion (xyzw) that represents a pure rotation of `deg`
    degrees around world‑Z axis."""
    from scipy.spatial.transform import Rotation as R
    return R.from_euler('z', deg, degrees=True).as_quat()


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # ========== Environment Setup ==========
    env, task = setup_environment()

    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step / observation for video capture
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ========== Retrieve all object positions ==========
        positions = get_object_positions()     # Dict[str, np.ndarray]

        def pos(name: str) -> np.ndarray:
            real_name = _alias_name(name)
            if real_name not in positions:
                raise KeyError(f"Object/position '{real_name}' not found in scene.")
            return np.array(positions[real_name], dtype=np.float32)

        # ------------------------------------------------------------------
        #  Oracle Plan Execution  (see specification “steps” array)
        # ------------------------------------------------------------------

        done = False
        reward = 0.0

        # Step‑1  rotate gripper to ninety_deg  (about Z)
        if not done:
            target_quat = _quat_about_z(90.0)
            obs, reward, done = rotate(env, task, target_quat)

        # Step‑2  move → side‑pos‑bottom
        if not done:
            obs, reward, done = move(env, task, pos("side-pos-bottom"))

        # Step‑3  move → anchor‑pos‑bottom
        if not done:
            obs, reward, done = move(env, task, pos("anchor-pos-bottom"))

        # Step‑4  pick the drawer handle located at anchor‑pos‑bottom
        #         (approach from +z direction keeps gripper above handle)
        if not done:
            obs, reward, done = pick(env, task,
                                     target_pos=pos("anchor-pos-bottom"),
                                     approach_axis='z')

        # Step‑5  pull the drawer  (pull 0.20 m along +x axis)
        if not done:
            obs, reward, done = pull(env, task,
                                     pull_distance=0.20,
                                     pull_axis='x')

        # Step‑6  move gripper away to a safe “nowhere” position
        if not done:
            obs, reward, done = move(env, task, pos("nowhere-pos"))

        # Step‑7  rotate gripper back to zero_deg
        if not done:
            obs, reward, done = rotate(env, task, _quat_about_z(0.0))

        # ------------------------------------------------------------------
        #  Move tomatoes (item1 / item2) onto the plate
        # ------------------------------------------------------------------

        tomatoes = ["tomato1", "tomato2"]
        plate_position = pos("plate")

        for tomato in tomatoes:
            if done:
                break

            # pick tomato on the table
            try:
                tomato_pos = pos(tomato)
            except KeyError:
                print(f"[Warning] {tomato} not found, skipping.")
                continue

            obs, reward, done = pick(env, task,
                                     target_pos=tomato_pos,
                                     approach_axis='z')
            if done:
                break

            # place tomato onto the plate
            obs, reward, done = place(env, task,
                                      target_pos=plate_position,
                                      approach_axis='z')
            if done:
                break

        # ------------------------------------------------------------------
        #  Final status
        # ------------------------------------------------------------------
        if done:
            print(f"[Result] Task finished early with reward {reward}.")
        else:
            print("[Result] Oracle plan executed completely.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
