import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# Import every predefined skill exactly as‑is
from skill_code import rotate, move, pull, pick, place


def run_skeleton_task():
    """Run the oracle plan that: (1) opens an unlocked drawer, 
       (2) moves two tomatoes onto a plate."""
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------
    #  Environment / task initialisation
    # ------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optionally start video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)           # wrap step
        task.get_observation = recording_get_observation(task.get_observation)

        # --------------------------------------------------------
        #  Lookup the required positions from the helper module
        # --------------------------------------------------------
        positions = get_object_positions()

        try:
            side_pos          = positions['bottom_side_pos']
            anchor_pos        = positions['bottom_anchor_pos']
            plate_pos         = positions['plate']
            tomato1_pos       = positions.get('tomato1', positions.get('item1'))
            tomato2_pos       = positions.get('tomato2', positions.get('item2'))
        except KeyError as e:
            raise RuntimeError(f"Missing object position for: {e.args[0]}")

        # --------------------------------------------------------
        #  Execute the oracle plan (Specification steps 1‑8)
        # --------------------------------------------------------

        # STEP‑1: rotate gripper to ninety_deg about Z axis
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Plan] Task finished unexpectedly after rotate.")
            return

        # STEP‑2: move to side‑pos‑bottom
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Plan] Task finished unexpectedly after move‑to‑side.")
            return

        # STEP‑3: move to anchor‑pos‑bottom (align with drawer handle)
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Plan] Task finished unexpectedly after move‑to‑anchor.")
            return

        # STEP‑4: pull the drawer open (0.20 m along +x)
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Plan] Task finished unexpectedly after pull.")
            return

        # --------------------------------------------------------
        #  Pick & place the first tomato
        # --------------------------------------------------------
        print("[Plan] Picking first tomato …")
        obs, reward, done = pick(env,
                                 task,
                                 target_pos=np.array(tomato1_pos),
                                 approach_distance=0.15,
                                 approach_axis='z')
        if done:
            print("[Plan] Task finished unexpectedly after first pick.")
            return

        print("[Plan] Placing first tomato on plate …")
        obs, reward, done = place(env,
                                  task,
                                  target_pos=np.array(plate_pos),
                                  approach_distance=0.15,
                                  approach_axis='z')
        if done:
            print("[Plan] Task finished unexpectedly after first place.")
            return

        # --------------------------------------------------------
        #  Pick & place the second tomato
        # --------------------------------------------------------
        print("[Plan] Picking second tomato …")
        obs, reward, done = pick(env,
                                 task,
                                 target_pos=np.array(tomato2_pos),
                                 approach_distance=0.15,
                                 approach_axis='z')
        if done:
            print("[Plan] Task finished unexpectedly after second pick.")
            return

        print("[Plan] Placing second tomato on plate …")
        obs, reward, done = place(env,
                                  task,
                                  target_pos=np.array(plate_pos),
                                  approach_distance=0.15,
                                  approach_axis='z')
        if done:
            print("[Plan] Task ended after successful completion!")
        else:
            print("[Plan] Goal executed but environment not signalling done. Check success criteria.")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()