import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Pull in every pre‑defined low‑level skill
from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

# Utility that returns a dict {object_name: np.ndarray([x, y, z])}
from object_positions import get_object_positions


def run_skeleton_task():
    """Execute the oracle plan given in the specification."""
    print("===== Starting Skeleton Task =====")

    # --------------------------------------------------------
    # 1)  Environment setup & book‑keeping for video recording
    # --------------------------------------------------------
    env, task = setup_environment()
    try:
        _, obs = task.reset()

        # (optional) video writers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # --------------------------------------------------------
        # 2)  Gather all positions we will need in the plan
        # --------------------------------------------------------
        positions = get_object_positions()

        # Drawer related positions
        pos_side_bottom   = positions.get('bottom_side_pos')
        pos_anchor_bottom = positions.get('bottom_anchor_pos')

        # Tomatoes (item1 / item2) and plate
        pos_tomato1 = positions.get('item1')
        pos_tomato2 = positions.get('item2')
        pos_plate   = positions.get('plate')

        # Sanity‑check that all positions actually exist
        required_keys = {
            'bottom_side_pos': pos_side_bottom,
            'bottom_anchor_pos': pos_anchor_bottom,
            'item1': pos_tomato1,
            'item2': pos_tomato2,
            'plate': pos_plate,
        }
        for k, v in required_keys.items():
            if v is None:
                raise KeyError(
                    f"[Task] Mandatory object position '{k}' was not found in "
                    f"get_object_positions().  Aborting execution."
                )

        # --------------------------------------------------------
        # 3)  Execute the oracle plan, step‑by‑step
        # --------------------------------------------------------
        done = False

        # Step‑1  Rotate gripper to “ninety_deg”
        if not done:
            print("\n--- [Plan Step 1] rotate(gripper, ninety_deg) ---")
            quat_90deg_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat=quat_90deg_z)

        # Step‑2  Move to the drawer side position
        if not done:
            print("\n--- [Plan Step 2] move(gripper, side-pos-bottom) ---")
            obs, reward, done = move(env, task, target_pos=pos_side_bottom)

        # Step‑3  Move to the drawer anchor (handle) position
        if not done:
            print("\n--- [Plan Step 3] move(gripper, anchor-pos-bottom) ---")
            obs, reward, done = move(env, task, target_pos=pos_anchor_bottom)

        # Step‑4  Pull the drawer open
        if not done:
            print("\n--- [Plan Step 4] pull(gripper, bottom) ---")
            # Heuristic: pull straight backwards along the robot +x axis by 0.12 m
            obs, reward, done = pull(env, task, pull_distance=0.12, pull_axis='x')

        # Step‑5  Pick first tomato (item1) from table
        if not done:
            print("\n--- [Plan Step 5] pick(item1, table) ---")
            obs, reward, done = pick(env, task, target_pos=pos_tomato1,
                                     approach_distance=0.15, approach_axis='z')

        # Step‑6  Place first tomato on the plate
        if not done:
            print("\n--- [Plan Step 6] place(item1, plate) ---")
            obs, reward, done = place(env, task, target_pos=pos_plate,
                                      approach_distance=0.15, approach_axis='z')

        # Step‑7  Pick second tomato (item2) from table
        if not done:
            print("\n--- [Plan Step 7] pick(item2, table) ---")
            obs, reward, done = pick(env, task, target_pos=pos_tomato2,
                                     approach_distance=0.15, approach_axis='z')

        # Step‑8  Place second tomato on the plate
        if not done:
            print("\n--- [Plan Step 8] place(item2, plate) ---")
            obs, reward, done = place(env, task, target_pos=pos_plate,
                                      approach_distance=0.15, approach_axis='z')

        # --------------------------------------------------------
        # 4)  Final state summary
        # --------------------------------------------------------
        if done:
            print("\n===== Task finished early (done=True) =====")
        else:
            print("\n===== Plan executed completely (done=False) =====")

    except Exception as ex:
        print(f"[Task] Exception encountered during execution: {ex}")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
