import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def _axis_name_from_vector(vec: np.ndarray) -> str:
    """Return the axis name (‘x’, ‘-x’, …) that best matches the given vector."""
    axis = np.array(vec, dtype=float)
    if np.linalg.norm(axis) == 0:
        return 'x'          # arbitrary, will never actually be used
    axis = axis / np.linalg.norm(axis)
    idx = np.argmax(np.abs(axis))          # dominant component
    sign = '' if axis[idx] >= 0 else '-'
    return f"{sign}{'xyz'[idx]}"


def run_skeleton_task() -> None:
    """Run the task: open an available drawer then move two tomatoes onto the plate."""
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        # ------------------------------------------------------------------
        # Reset & wrap for video
        # ------------------------------------------------------------------
        descriptions, obs = task.reset()
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------
        # Query all useful positions in the scene
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Drawer that is not locked -------------------------------------------------
        drawer_side      = positions['bottom_side_pos']
        drawer_anchor    = positions['bottom_anchor_pos']
        drawer_joint     = positions['bottom_joint_pos']       # for figuring out pull axis
        # ---------------------------------------------------------------------------
        item_names = ['item1', 'item2']      # we only need two tomatoes
        plate_pos  = positions['plate']

        # ------------------------------------------------------------------
        # 1) rotate  (zero_deg -> ninety_deg)
        # ------------------------------------------------------------------
        current_quat = np.array(obs.gripper_pose[3:7])
        ninety_quat  = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()
        print("[PLAN] Step 1 : rotate to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[TASK] Ended unexpectedly after rotate.")
            return

        # ------------------------------------------------------------------
        # 2) move  : approach drawer side position
        # ------------------------------------------------------------------
        print("[PLAN] Step 2 : move to bottom_side_pos ->", drawer_side)
        obs, reward, done = move(env, task, drawer_side)
        if done:
            print("[TASK] Ended unexpectedly after move to side.")
            return

        # ------------------------------------------------------------------
        # 3) move  : move to anchor (handle) position
        # ------------------------------------------------------------------
        print("[PLAN] Step 3 : move to bottom_anchor_pos ->", drawer_anchor)
        obs, reward, done = move(env, task, drawer_anchor)
        if done:
            print("[TASK] Ended unexpectedly after move to anchor.")
            return

        # ------------------------------------------------------------------
        # 4) pick  : grip the drawer handle
        # ------------------------------------------------------------------
        print("[PLAN] Step 4 : pick drawer handle at anchor.")
        obs, reward, done = pick(env, task, target_pos=drawer_anchor,
                                 approach_distance=0.10, approach_axis='z')
        if done:
            print("[TASK] Ended unexpectedly after pick handle.")
            return

        # ------------------------------------------------------------------
        # 5) pull  : pull the drawer open
        # ------------------------------------------------------------------
        pull_vec   = drawer_anchor - drawer_joint
        pull_axis  = _axis_name_from_vector(pull_vec)
        pull_dist  = np.linalg.norm(pull_vec) + 0.10   # add small extra distance
        print(f"[PLAN] Step 5 : pull drawer along {pull_axis} by {pull_dist:.3f} m")
        obs, reward, done = pull(env, task, pull_distance=pull_dist, pull_axis=pull_axis)
        if done:
            print("[TASK] Ended unexpectedly after pull.")
            return

        # ==================================================================
        # With the drawer open we now manipulate the tomatoes/items.
        # ==================================================================
        for idx, item_name in enumerate(item_names, start=1):
            item_pos = positions[item_name]
            # 6 / 10) move to item
            print(f"[PLAN] Step {5 + (idx-1)*4 + 1} : move to {item_name} ->", item_pos)
            obs, reward, done = move(env, task, item_pos)
            if done:
                print(f"[TASK] Ended unexpectedly while moving to {item_name}.")
                return

            # 7 / 11) pick item
            print(f"[PLAN] Step {5 + (idx-1)*4 + 2} : pick {item_name}")
            obs, reward, done = pick(env, task, target_pos=item_pos,
                                     approach_distance=0.10, approach_axis='z')
            if done:
                print(f"[TASK] Ended unexpectedly after picking {item_name}.")
                return

            # 8 / 12) move towards plate
            print(f"[PLAN] Step {5 + (idx-1)*4 + 3} : move to plate ->", plate_pos)
            obs, reward, done = move(env, task, plate_pos)
            if done:
                print("[TASK] Ended unexpectedly while moving to plate.")
                return

            # 9 / 13) place item on plate
            print(f"[PLAN] Step {5 + (idx-1)*4 + 4} : place {item_name} on plate")
            obs, reward, done = place(env, task, target_pos=plate_pos,
                                      approach_distance=0.10, approach_axis='z')
            if done:
                print("[TASK] Completed early while placing on plate.")
                break  # Successful termination

        print("[TASK] All steps executed. Task should be complete!")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
