import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from skill_code import rotate, move, pick, place, pull


# -----------------------------------------------------------------------------#
# Helper utilities                                                             #
# -----------------------------------------------------------------------------#
def _lookup_position(obj_name, fallback_in_world=True):
    """
    Utility that tries to obtain the 3‑D position of an object.  
    1) First consult the dictionary returned by `object_positions.get_object_positions()`.  
    2) If the name is missing (or None) and `fallback_in_world` is True, fall back
       to asking PyRep directly via `Shape(obj_name).get_position()`.
    """
    try:
        pos = ALL_POSITIONS.get(obj_name, None)
        if pos is not None:
            return np.asarray(pos, dtype=float)
        if fallback_in_world:
            return np.asarray(Shape(obj_name).get_position(), dtype=float)
    except Exception as e:
        print(f"[Warning] Could not fetch position for '{obj_name}': {e}")
    return None


def _plan_step(label, fn, *args, **kwargs):
    """Convenience wrapper that executes one skill and prints uniform logs."""
    print(f"\n===== [Plan] {label} =====")
    try:
        obs, reward, done = fn(*args, **kwargs)
        return obs, reward, done
    except Exception as exc:
        print(f"[Error] Step '{label}' failed with exception: {exc}")
        raise


# -----------------------------------------------------------------------------#
# Main orchestration                                                           #
# -----------------------------------------------------------------------------#
def run_skeleton_task():
    print("===========  Task: Open Drawer + Move Tomatoes  ===========")

    # === Environment bootstrap ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)          # wrap step
        task.get_observation = recording_get_observation(task.get_observation)

        # --- Gather object positions once at the beginning ---
        global ALL_POSITIONS       # Needed inside helper
        ALL_POSITIONS = get_object_positions()

        # ------------------------------------------------------------------ #
        #  Oracle plan (specification)                                       #
        # ------------------------------------------------------------------ #

        # STEP‑1  rotate(gripper, zero_deg → ninety_deg)
        ninety_z_quat = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = _plan_step(
            "Rotate gripper to 90°",
            rotate, env, task,
            target_quat=ninety_z_quat
        )
        if done:
            print("[Task] Finished immediately after rotate.")
            return

        # STEP‑2  move‑to‑side(gripper, bottom, nowhere_pos → side-pos-bottom)
        side_pos = _lookup_position('bottom_side_pos')
        if side_pos is None:
            raise RuntimeError("Could not resolve 'bottom_side_pos'.")
        obs, reward, done = _plan_step(
            "Move to drawer side position",
            move, env, task, target_pos=side_pos
        )
        if done:
            return

        # STEP‑3  move‑to‑anchor(gripper, bottom, side‑pos → anchor‑pos)
        anchor_pos = _lookup_position('bottom_anchor_pos')
        if anchor_pos is None:
            raise RuntimeError("Could not resolve 'bottom_anchor_pos'.")
        obs, reward, done = _plan_step(
            "Move to drawer anchor position",
            move, env, task, target_pos=anchor_pos
        )
        if done:
            return

        # STEP‑4  pick‑drawer(gripper, bottom, anchor‑pos)
        #         (Use generic `pick` to close the gripper on the handle.)
        obs, reward, done = _plan_step(
            "Grasp drawer handle",
            pick, env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,   # shorter distance for precise grasp
            approach_axis='-z'
        )
        if done:
            return

        # STEP‑5  pull(gripper, bottom)
        #         (Positive X‑axis pull of ~0.20 m; adjust if necessary.)
        obs, reward, done = _plan_step(
            "Pull drawer open",
            pull, env, task,
            pull_distance=0.20,
            pull_axis='x'
        )
        if done:
            return

        # -----------------  HANDLE TOMATO 1  -----------------
        tomato1_pos = _lookup_position('tomato1')
        plate_pos   = _lookup_position('plate')
        if tomato1_pos is None or plate_pos is None:
            raise RuntimeError("Could not obtain positions for tomato1/plate.")

        obs, reward, done = _plan_step(
            "Pick tomato1",
            pick, env, task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            approach_axis='-z'
        )
        if done:
            return

        obs, reward, done = _plan_step(
            "Place tomato1 on plate",
            place, env, task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis='-z'
        )
        if done:
            return

        # -----------------  HANDLE TOMATO 2  -----------------
        tomato2_pos = _lookup_position('tomato2')
        if tomato2_pos is None:
            raise RuntimeError("Could not obtain position for tomato2.")

        obs, reward, done = _plan_step(
            "Pick tomato2",
            pick, env, task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            approach_axis='-z'
        )
        if done:
            return

        obs, reward, done = _plan_step(
            "Place tomato2 on plate",
            place, env, task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis='-z'
        )
        # If the environment signals done we simply exit; otherwise we end
        # gracefully after finishing the plan.
        if done:
            print("[Task] Completed with 'done=True'.")
        else:
            print("[Task] Plan executed. Goal should be satisfied.")

    finally:
        shutdown_environment(env)
        print("===========  Task Finished  ===========")



if __name__ == "__main__":
    run_skeleton_task()
