# run_skeleton_task.py  (Completed Implementation)

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Pull every primitive in skill_code into the local namespace
from skill_code import *          # noqa: F401,F403

from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

from object_positions import get_object_positions


def run_skeleton_task():
    """
    Execute the oracle plan that (1) opens the bottom drawer and
    (2) moves both tomatoes onto the plate.
    All low‑level work is done by the predefined skill functions coming from
    `skill_code`.
    """
    print("===== Starting Skeleton Task =====")

    # ==============================================================
    #  Environment setup & reset
    # ==============================================================
    env, task = setup_environment()
    try:
        # Reset task to initial state
        descriptions, obs = task.reset()

        # ----------------------------------------------------------
        # Optional: video recorder helpers
        # ----------------------------------------------------------
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ==========================================================
        #  Retrieve object positions for everything we will touch
        # ==========================================================
        positions = get_object_positions()

        # Helper utility for safe dictionary access
        def p(name):
            if name not in positions:
                raise KeyError(
                    f"[run_skeleton_task] Expected '{name}' in "
                    f"object_positions, but it was not found."
                )
            return positions[name]

        # Drawer‑related key points
        nowhere_pos        = p("waypoint1")              # starting “parking” position
        bottom_side_pos    = p("bottom_side_pos")
        bottom_anchor_pos  = p("bottom_anchor_pos")

        # Tomatoes & plate
        tomato1_pos        = p("tomato1")
        tomato2_pos        = p("tomato2")
        plate_pos          = p("plate")

        # ==========================================================
        #  Step‑by‑step execution of the oracle plan
        # ==========================================================
        done = False
        reward = 0.0

        # ---------- Step 1  : rotate(gripper, zero_deg, ninety_deg)
        # Build a target quaternion that represents +90° about the Z‑axis.
        target_quat = R.from_euler("z", 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Terminated after step 1.")
            return

        # ---------- Step 2  : move‑to‑side(gripper, bottom, nowhere‑pos, side‑pos‑bottom)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Terminated after step 2.")
            return

        # ---------- Step 3  : move‑to‑anchor(gripper, bottom, side‑pos‑bottom, anchor‑pos‑bottom)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Terminated after step 3.")
            return

        # ---------- Step 4  : pick‑drawer(gripper, bottom, anchor‑pos‑bottom)
        # We approximate drawer‑handle grasping with the generic `pick` skill,
        # targeting the anchor position we just moved to.
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Terminated after step 4.")
            return

        # ---------- Step 5  : pull(gripper, bottom)
        # Pull straight out along the X‑axis by 15 cm to open the drawer.
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis="x")
        if done:
            print("[Task] Terminated after step 5.")
            return

        # Optional: back away a little so the drawer doesn’t collide when we
        # move the arm above the tomatoes.
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Terminated while backing away from drawer.")
            return

        # ========== Tomato 1 ==========
        # ---------- Step 6  : pick(tomato1, table)
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Terminated after picking tomato1.")
            return

        # ---------- Step 7  : place(tomato1, plate)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Terminated after placing tomato1.")
            return

        # ========== Tomato 2 ==========
        # ---------- Step 8  : pick(tomato2, table)
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Terminated after picking tomato2.")
            return

        # ---------- Step 9  : place(tomato2, plate)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Terminated after placing tomato2.")
            return

        # ----------------------------------------------------------
        #  Task finished — provide a short summary
        # ----------------------------------------------------------
        print("===== Task completed successfully! Reward:", reward)

    except Exception as e:
        # Generic safety net – print the error so the user sees it,
        # but always shut down the simulator in `finally`.
        print("[run_skeleton_task] Exception occurred:", e)
        raise

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()