import numpy as np
from scipy.spatial.transform import Rotation as R
import traceback

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# ===== Do NOT redefine any skills – import the predefined ones =====
from skill_code import move, pick, place, rotate, pull


"""
[Frozen Code Start]
    obs, reward, done = rotate(env, task, target_quat=quat_90deg_z)
obs, reward, done = move(env, task, target_pos=side_pos)
obs, reward, done = move(env, task, target_pos=anchor_pos)
obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_axis="-z"    # descend onto the handle
        )
obs, reward, done = pull(
            env, task,
            pull_distance=0.15,    # 15 cm pull
            pull_axis="x"
        )
[Frozen Code End]
"""


def run_skeleton_task():
    """
    Execute the high-level oracle plan:

        1) Rotate gripper from zero_deg to ninety_deg.
        2) Move gripper to the drawer’s side position.
        3) Move gripper to the drawer’s anchor (handle) position.
        4) Pick / grasp the drawer handle.
        5) Pull the drawer open.
        6) Pick tomato 1 from the table, place on the plate.
        7) Pick tomato 2 from the table, place on the plate.
    """
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()

    try:
        # ------------------------------------------------------------
        #  Reset task & start video recording
        # ------------------------------------------------------------
        _, obs = task.reset()
        init_video_writers(obs)

        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------
        #  Fetch required object positions
        # ------------------------------------------------------------
        positions = get_object_positions()

        required_keys = [
            "bottom_side_pos", "bottom_anchor_pos",
            "tomato1", "tomato2", "plate"
        ]
        for key in required_keys:
            if key not in positions:
                raise KeyError(f"[Task] Missing key in object positions dict: '{key}'")

        # Cast to numpy arrays for convenience
        side_pos    = np.asarray(positions["bottom_side_pos"],  dtype=float)
        anchor_pos  = np.asarray(positions["bottom_anchor_pos"], dtype=float)
        tomato1_pos = np.asarray(positions["tomato1"],          dtype=float)
        tomato2_pos = np.asarray(positions["tomato2"],          dtype=float)
        plate_pos   = np.asarray(positions["plate"],            dtype=float)

        # ============================================================
        #  [Step 1] Rotate gripper 90° about Z
        # ============================================================
        quat_90deg_z = R.from_euler("z", 90, degrees=True).as_quat()  # xyzw

        # ------------------------------------------------------------
        #  [Frozen] Rotate
        # ------------------------------------------------------------
        obs, reward, done = rotate(env, task, target_quat=quat_90deg_z)
        if done:
            print("[Task] Finished early after rotate.")
            return

        # ------------------------------------------------------------
        #  [Frozen] Move to drawer side position
        # ------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Finished early while moving to drawer side.")
            return

        # ------------------------------------------------------------
        #  [Frozen] Move to drawer anchor / handle
        # ------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Finished early while moving to drawer anchor.")
            return

        # ------------------------------------------------------------
        #  [Frozen] Pick the drawer handle (approach from above)
        # ------------------------------------------------------------
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_axis="-z"    # descend onto the handle
        )
        if done:
            print("[Task] Finished early while grasping drawer handle.")
            return

        # ------------------------------------------------------------
        #  [Frozen] Pull to slide the drawer open
        # ------------------------------------------------------------
        obs, reward, done = pull(
            env, task,
            pull_distance=0.15,    # 15 cm pull
            pull_axis="x"
        )
        if done:
            print("[Task] Finished early after pulling drawer open.")
            return

        # ============================================================
        #  Helper: lift above tomatoes before descending
        # ============================================================
        def hover_above(pos, height=0.10):
            return pos + np.array([0.0, 0.0, height], dtype=float)

        # ============================================================
        #  [Step 6] Pick tomato 1
        # ============================================================
        obs, reward, done = move(env, task, target_pos=hover_above(tomato1_pos))
        if done:
            print("[Task] Ended unexpectedly while moving above tomato1.")
            return

        obs, reward, done = pick(
            env, task,
            target_pos=tomato1_pos,
            approach_axis="-z"
        )
        if done:
            print("[Task] Ended unexpectedly while picking tomato1.")
            return

        # ============================================================
        #  [Step 7] Place tomato 1 onto plate
        # ============================================================
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_axis="-z"
        )
        if done:
            print("[Task] Ended unexpectedly after placing tomato1.")
            return

        # ============================================================
        #  [Step 8] Pick tomato 2
        # ============================================================
        obs, reward, done = move(env, task, target_pos=hover_above(tomato2_pos))
        if done:
            print("[Task] Ended unexpectedly while moving above tomato2.")
            return

        obs, reward, done = pick(
            env, task,
            target_pos=tomato2_pos,
            approach_axis="-z"
        )
        if done:
            print("[Task] Ended unexpectedly while picking tomato2.")
            return

        # ============================================================
        #  [Step 9] Place tomato 2 onto plate
        # ============================================================
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_axis="-z"
        )

        # ------------------------------------------------------------
        #  Final outcome
        # ------------------------------------------------------------
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Plan executed; environment reports done = False.")

    except Exception as exc:
        print("=== Exception during task execution ===")
        traceback.print_exc()
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()