# run_skeleton_task.py (Completed Implementation)

import re
import numpy as np
from pyrep.objects.shape import Shape              # (kept – do not remove)
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *                            # pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ---------------------------------------------------------------------------
# Helper utilities
# ---------------------------------------------------------------------------
_COMBINED_DOMAIN_STR = """
(define (domain combined-domain)
  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)
  (:types object location drawer - object gripper - object position - object angle - object)
  (:predicates
    (at ?obj - object ?loc - location)
    (holding ?obj - object)
    (handempty)
    (is-locked ?d - drawer)
    (is-open ?d - drawer)
    (rotated ?g - gripper ?a - angle)
    (gripper-at ?g - gripper ?p - position)
    (holding-drawer ?g - gripper ?d - drawer)
    (is-side-pos ?p - position ?d - drawer)
    (is-anchor-pos ?p - position ?d - drawer)
  )
)
"""

_EXPLORATION_DOMAIN_STR = """
(define (domain exploration)
  (:requirements :strips :typing :conditional-effects :universal-preconditions)
  (:types robot object location)
  (:predicates
    (robot-at ?r - robot ?loc - location)
    (at ?obj - object ?loc - location)
    (identified ?obj - object)
    (temperature-known ?obj - object)
    (holding ?obj - object)
    (handempty)
    (weight-known ?obj - object)
    (durability-known ?obj - object)
    (lock-known ?obj - object)
  )
)
"""


def _extract_predicates(pddl_str: str) -> set:
    """Extract predicate names from a PDDL string."""
    section = re.search(r'\(:predicates(.*?)\)', pddl_str, re.S | re.I)
    if not section:
        return set()
    # Look for tokens that start with '(' and are followed by a symbol
    tokens = re.findall(r'\(([^ \t\r\n\()]+)', section.group(1))
    return set(tokens)


def find_missing_predicates() -> set:
    """Compare combined-domain and exploration-domain, return missing predicates."""
    combined_preds = _extract_predicates(_COMBINED_DOMAIN_STR)
    exploration_preds = _extract_predicates(_EXPLORATION_DOMAIN_STR)
    return exploration_preds - combined_preds


# ---------------------------------------------------------------------------
# Main task runner
# ---------------------------------------------------------------------------
def run_skeleton_task():
    """Generic skeleton for running any task in your simulation (completed)."""
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Optionally record the task
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------
        # STEP 1 – Static predicate analysis: find what is missing
        # ------------------------------------------------------------------
        missing_predicates = find_missing_predicates()
        if missing_predicates:
            print(f"[Analysis] Missing predicates relative to combined domain: {missing_predicates}")
        else:
            print("[Analysis] No missing predicates found between domains.")

        # ------------------------------------------------------------------
        # STEP 2 – Simple skill-based exploration of environment
        # ------------------------------------------------------------------
        positions = get_object_positions()      # dictionary:  {name: (x, y, z)}
        if not positions:
            print("[Exploration] No objects returned from get_object_positions().")
        else:
            print(f"[Exploration] Retrieved {len(positions)} objects for exploration.")

            # Attempt to pick the first reachable object (generic exploration)
            first_obj_name, first_obj_pos = next(iter(positions.items()))
            print(f"[Exploration] Attempting to move to and pick '{first_obj_name}' at {first_obj_pos}.")

            # ---- Move phase ------------------------------------------------
            try:
                # Signature of `move` may vary across skill implementations.
                # The most common RLBench helper uses (env, task, target_pos, ...)
                move(env, task, target_pos=first_obj_pos)
                print(f"[Exploration] Move to '{first_obj_name}' succeeded.")
            except Exception as e:
                print(f"[Exploration] Move skill raised an exception: {e}")

            # ---- Pick phase ------------------------------------------------
            try:
                pick(env, task, target_pos=first_obj_pos)
                print(f"[Exploration] Pick '{first_obj_name}' succeeded.")
            except Exception as e:
                print(f"[Exploration] Pick skill raised an exception: {e}")

            # ---- Place phase (put the object back roughly where it was) ----
            try:
                place(env, task, target_pos=first_obj_pos)
                print(f"[Exploration] Place '{first_obj_name}' succeeded.")
            except Exception as e:
                print(f"[Exploration] Place skill raised an exception: {e}")

        # ------------------------------------------------------------------
        # STEP 3 – Use another predefined skill (rotate) just to demonstrate
        # ------------------------------------------------------------------
        try:
            identity_quat = np.array([0.0, 0.0, 0.0, 1.0])  # xyzw
            rotate(env, task, target_quat=identity_quat, max_steps=50, threshold=0.02)
        except Exception as e:
            print(f"[Rotate] Skill raised an exception: {e}")

        # ------------------------------------------------------------------
        # STEP 4 – Optional drawer interaction (pull) when available
        # ------------------------------------------------------------------
        try:
            # If a drawer handle exists in the object list, attempt to pull.
            drawer_candidates = [name for name in positions.keys() if "drawer" in name.lower()]
            if drawer_candidates:
                drawer_name = drawer_candidates[0]
                drawer_pos = positions[drawer_name]
                print(f"[Drawer] Found drawer candidate '{drawer_name}' at {drawer_pos}. Attempting interaction.")
                move(env, task, target_pos=drawer_pos)
                pull(env, task)             # pull signature often (env, task, ...)
                print("[Drawer] Pull action executed.")
            else:
                print("[Drawer] No drawer object detected in the scene.")
        except Exception as e:
            print(f"[Drawer] Pull skill raised an exception: {e}")

        # ------------------------------------------------------------------
        # Exploration complete
        # ------------------------------------------------------------------
        print("===== Exploration Phase Complete =====")

    finally:
        # Always ensure the environment is properly shut down
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()