# run_skeleton_task.py  (Completed)

import sys
import time
import traceback
from typing import Dict, Any

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# -------------------------------------------------
# All primitive robot skills live inside skill_code. 
# DO NOT re-define any of them – just import and use.
# -------------------------------------------------
from skill_code import move, pick, place, rotate, pull   # noqa: F401  (imported for side-effect / use)

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -------------------------------------------------
# Utility helpers
# -------------------------------------------------
def pprint_positions(msg: str, pos_dict: Dict[str, Any]) -> None:
    """Nicely print a dictionary of object positions (x,y,z)."""
    print(f"{msg}:")
    for k, v in pos_dict.items():
        print(f"   • {k:20s} -> {np.array(v)}")


def ensure_handempty(env, task):
    """
    Very small exploration routine whose *sole* purpose is to make sure that the
    predicate ‘handempty’ is (logically) true in the simulated world.  The only
    way we can influence that predicate with the predefined skills is via
    pick/place – placing any currently-held object will re-establish handempty.

    Implementation strategy
    -----------------------
    1) Query the task’s observation to guess whether something is being held.
       RLBench encodes that information per-task; the *most* generic signal
       available is ‘gripper_open’ or ‘gripper_openness’.  We look for both.
    2) If the gripper looks *closed* we *assume* we are holding something and
       attempt to drop it directly underneath the gripper (no need to search
       for a receptacle; we only care about handempty).
    3) If the guess says the gripper is open, we do nothing – handempty is
       already satisfied.

    Note
    ----
    We do not create any new skill primitives – we only combine the existing
    ‘move’ and ‘place’ skills in a defensive manner.
    """
    print("---- [Exploration] Ensuring the predicate `handempty` is satisfied.")
    obs = task.get_observation()

    # Step-1  — heuristic check
    gripper_open_attr = None
    for cand in ["gripper_open", "gripper_openness", "gripper_open_amount"]:
        if hasattr(obs, cand):
            gripper_open_attr = cand
            break

    holding_suspected = False
    if gripper_open_attr is not None:
        open_val = getattr(obs, gripper_open_attr)
        # Convention in RLBench: +1 = open, -1 = closed
        if isinstance(open_val, (list, tuple, np.ndarray)):
            open_val = float(open_val[0])
        holding_suspected = open_val < 0   # closed → maybe holding
        print(f"      · Found attribute '{gripper_open_attr}'={open_val:.3f} ➜ "
              f"{'closed' if holding_suspected else 'open'} gripper.")
    else:
        # Fallback – no attribute → play safe and do nothing.
        print("      · Could not find a gripper-open attribute; skipping check.")
        return

    # Step-2  — if holding, place object directly beneath current pose
    if holding_suspected:
        print("      · Gripper seems closed.  Attempting to *place* the object to regain handempty.")
        cur_pose = obs.gripper_pose
        drop_pos = np.array(cur_pose[:3]) + np.array([0.0, 0.0, -0.10])  # 10 cm below
        try:
            # First move slightly up to avoid collisions, then down
            safe_pos = np.array(cur_pose[:3]) + np.array([0.0, 0.0, 0.08])
            move(env, task, safe_pos)
            move(env, task, drop_pos)

            # The place() API generally needs an *object id*.  We do not know it here
            # (this is a generic exploration routine).  Many RLBench tasks implement
            # place() such that `obj_name` is ignored / inferred from the internal
            # grasped-object handle.  We therefore pass `None` and fall back to
            # whatever the underlying skill does.  If the skill requires an object
            # name, it will raise and we handle the exception gracefully.
            try:
                obs, _, _ = place(env, task, target_pos=drop_pos)
                print("      · Place succeeded.")
            except Exception as e:
                print("      · Place() failed (interface mismatch).  "
                      "Attempting to OPEN gripper manually as last resort.")
                # Build a low-level action that only toggles the gripper joint.
                # We do not have a dedicated ‘open’ skill; however, the last entry
                # of the action vector is *gripper open amount*.
                action = np.zeros(env.action_shape)
                action[:3] = drop_pos
                action[3:7] = obs.gripper_pose[3:7]   # keep current orientation
                action[-1] = +1.0                     # fully open
                task.step(action)

            # After the attempted drop, wait a couple simulation steps so that the
            # physics engine settles and the world updates ‘handempty’.
            for _ in range(5):
                action = np.zeros(env.action_shape)
                action[:3] = obs.gripper_pose[:3]
                action[3:7] = obs.gripper_pose[3:7]
                action[-1] = +1.0
                task.step(action)

        except Exception as e:
            print("      · Encountered exception while trying to regain handempty:")
            traceback.print_exc()
            print("      · Continuing – the predicate might still be false.")
    else:
        print("      · Gripper is open – handempty already satisfied.")


# -------------------------------------------------
# Main entry point
# -------------------------------------------------
def run_skeleton_task():
    """Generic task runner that now includes an *exploration* phase which
    resolves the missing predicate ‘handempty’ identified in prior feedback."""
    print("\n==============================================")
    print("=====         Starting Skeleton Task      =====")
    print("==============================================\n")

    env, task = setup_environment()          # Environment boot-up
    try:
        # ---- Reset task & obtain first observation
        descriptions, obs = task.reset()

        # ---- Optional video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # -------------------------------------------------
        # 1) Exploration — find & satisfy missing predicate
        # -------------------------------------------------
        ensure_handempty(env, task)

        # -------------------------------------------------
        # 2) Retrieve object positions (world knowledge)
        # -------------------------------------------------
        positions = get_object_positions()
        if positions:
            pprint_positions("[Info] Initial object positions", positions)
        else:
            print("[Warning] get_object_positions() returned an empty dict.")

        # Example: choose a *single* object (first dict entry) to demonstrate a
        # pick-and-place pipeline.  In a real solution the *oracle plan* would
        # dictate the exact sequence; here we simply illustrate the mechanics.
        if positions:
            some_obj, some_pos = next(iter(positions.items()))
            print(f"\n[Plan] Demonstration – will pick '{some_obj}' and place it 15 cm to the right.")

            target_pick_pos = np.array(some_pos)
            target_place_pos = target_pick_pos + np.array([0.15, 0.0, 0.0])

            # --- Step-1 : move above the object then pick it
            safe_above = np.array(target_pick_pos) + np.array([0.0, 0.0, 0.10])
            move(env, task, safe_above)
            obs, reward, done = pick(env, task, target_pos=target_pick_pos)

            if done:
                print("[Early-Exit] Task signalled done after pick().")
                return

            # --- Step-2 : move to place location and drop
            safe_above_place = target_place_pos + np.array([0.0, 0.0, 0.10])
            move(env, task, safe_above_place)
            move(env, task, target_place_pos)
            obs, reward, done = place(env, task, target_pos=target_place_pos)

            if done:
                print("[Early-Exit] Task signalled done after place().")
                return

            print("\n[Success] Demonstration pick-and-place finished.")

        else:
            print("[Info] No objects reported – skipping demonstration actions.")

        # -------------------------------------------------
        # Additional oracle plan steps would continue here
        # -------------------------------------------------

    except KeyboardInterrupt:
        print("\n[Abort] Keyboard interrupt – shutting down cleanly.")
    except Exception as e:
        print("\n[Error] Unhandled exception in run_skeleton_task():")
        traceback.print_exc()
    finally:
        # Ensure environment shutdown no matter what happened
        shutdown_environment(env)
        print("\n==============================================")
        print("=====      End of Skeleton Task           =====")
        print("==============================================\n")


if __name__ == "__main__":
    run_skeleton_task()
