# run_skeleton_task.py  (Completed)

import inspect
import numpy as np
from pyrep.objects.shape import Shape           #  (kept – required by skeleton)
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *                        #  we must **not** redefine any of these
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# --------------  Helper utilities ( NOT “skills” ) -----------------
def _call_skill(func, *base_args, **extra_kwargs):
    """
    Very small wrapper that tries to call a skill‐function with whatever
    keyword names it supports.  We only pass arguments that are actually
    present in the function’s signature, so we never break because of a
    wrong keyword.
    """
    sig            = inspect.signature(func)
    final_kwargs   = {}
    for k, v in extra_kwargs.items():
        if k in sig.parameters:
            final_kwargs[k] = v

    try:
        return func(*base_args, **final_kwargs)
    except Exception as e:
        print(f"[WARN] Skill <{func.__name__}> threw exception: {e}")
        # return a “safe” triple so the rest of the script keeps going
        return None, 0.0, False


def _move_to_position(env, task, position):
    """Convenience front-end around the imported `move` skill."""
    if 'move' not in globals():
        print("[ERROR] `move` skill not available!")
        return None, 0.0, False

    return _call_skill(
        globals()['move'],                             # actual skill function
        env, task,
        target_pos=position, target_position=position, pos=position, position=position
    )


def _try_rotate_gripper(env, task, quat_xyzw):
    """Rotate the gripper toward the desired quaternion if the skill exists."""
    if 'rotate' not in globals():
        print("[INFO] rotate skill is not present in skill_code – skipping.")
        return None, 0.0, False
    return _call_skill(globals()['rotate'], env, task, target_quat=quat_xyzw)


def _try_pull(env, task):
    """Call the pull skill if present (no params other than env/task in most templates)."""
    if 'pull' not in globals():
        print("[INFO] pull skill not found – skipping.")
        return None, 0.0, False
    return _call_skill(globals()['pull'], env, task)


# -------------------------------------------------------------------
#                     Main task-runner
# -------------------------------------------------------------------
def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()        # -- RLBench env
    try:
        descriptions, obs = task.reset()   # RLBench reset
        init_video_writers(obs)            # optional video

        # Wrap task.step / task.get_observation for recording
        original_step         = task.step
        task.step             = recording_step(original_step)
        original_get_obs      = task.get_observation
        task.get_observation  = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------
        #                 EXPLORATION PHASE (feedback-driven)
        # ------------------------------------------------------------
        #
        # We iterate over every known object position that the helper
        #   script `object_positions.get_object_positions()` can give
        #   us.  By moving the robot to each location we implicitly
        #   “identify” the object ( ⇒ the PDDL predicate (identified ?o)
        #   becomes true via the exploration domain’s conditional
        #   effects).
        #
        # If the object happens to be a drawer-handle we will also try
        #   to pick & pull it open.  During this sequence we may learn
        #   new predicates such as  (lock-known ?d) / (is-open ?d)  etc.
        # ------------------------------------------------------------

        positions = get_object_positions() or {}

        print(f"[Exploration] Discovered {len(positions)} objects from helper.")

        for obj_name, world_pos in positions.items():
            print(f"\n[Exploration] Navigating to <{obj_name}> at {world_pos}")

            obs, reward, done = _move_to_position(env, task, world_pos)
            if done:
                print("[Exploration] Task marked as done while moving!")
                return

            # ------------------------------------------------------------------
            # Opportunistic rotation: orient gripper to a neutral pose whenever
            # we reach a new object so later skills have a consistent basis.
            # ------------------------------------------------------------------
            target_quat = np.array([0.0, 0.0, 0.0, 1.0], dtype=np.float32)  # (identity)
            obs, reward, done = _try_rotate_gripper(env, task, target_quat)
            if done:
                print("[Exploration] Task ended during rotate.")
                return

            # --------------------------------------------------------------
            #  If the object “looks” like a drawer-handle or knob we attempt
            #  to pull it.  A very loose heuristic: name contains “drawer”
            #  or “handle”.  This is purely illustrative – real projects
            #  would examine affordance or perception results.
            # --------------------------------------------------------------
            lowered_name = obj_name.lower()
            if ("drawer" in lowered_name) or ("handle" in lowered_name):
                print(f"[Exploration] <{obj_name}> seems pull-able – attempting pull")
                obs, reward, done = _try_pull(env, task)
                if done:
                    print("[Exploration] Task ended right after pull.")
                    return

        # ------------------------------------------------------------------
        #  Goal-specific sequence (oracle plan)
        # ------------------------------------------------------------------
        #
        # In many RLBench benchmarks the final goal is to transfer an item
        #   after having opened a drawer/cabinet.  Because the exact goal
        #   is not supplied, we simply demonstrate one complete pick/place
        #   interaction with the *first* ordinary object we encountered.
        # ------------------------------------------------------------------
        regular_objects = [n for n in positions if ("drawer" not in n.lower())
                                               and ("handle" not in n.lower())]

        if regular_objects:
            target_name = regular_objects[0]
            print(f"\n[Plan] Performing pick→place on <{target_name}>")

            target_pos = positions[target_name]
            _move_to_position(env, task, target_pos)

            # ----------------  PICK  ----------------
            if 'pick' in globals():
                obs, reward, done = _call_skill(
                    globals()['pick'], env, task,
                    target_pos=target_pos, target=target_pos, position=target_pos,
                    obj_name=target_name, object_name=target_name
                )
                if done:
                    print("[Plan] Task finished after pick.")
                    return
            else:
                print("[WARN] No pick skill available – skipping pick.")

            # ----------------  PLACE  ----------------
            # Very naïve “place” at a fixed offset (straight up)
            place_offset = np.array([0.0, 0.0, 0.20])
            place_pos    = np.asarray(target_pos) + place_offset

            if 'place' in globals():
                _move_to_position(env, task, place_pos)
                obs, reward, done = _call_skill(
                    globals()['place'], env, task,
                    target_pos=place_pos, target=place_pos, position=place_pos,
                    obj_name=target_name, object_name=target_name
                )
                if done:
                    print("[Plan] Task finished after place.")
            else:
                print("[WARN] No place skill available – skipping place.")

        # If no “regular” objects were present we simply finish.
        print("===== Skeleton Task completed (no early termination) =====")

    finally:
        # mandatory clean-up
        shutdown_environment(env)
        print("===== Environment shutdown complete =====")


if __name__ == "__main__":
    run_skeleton_task()