# run_skeleton_task.py (Completed Executable)

import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import every predefined skill exactly as delivered in the framework
from skill_code import *      # noqa: F403, F401

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ------------------------------------------------------------
# Helper utilities – no new “skills”, merely helpers around the
# already–implemented primitives that ship with the framework.
# ------------------------------------------------------------
def _is_hand_empty(task, empty_threshold: float = 0.9) -> bool:
    """
    Quick-and-dirty test that tries to infer whether the gripper
    is free.  All RLBench tasks expose `gripper_open_amount`
    inside the observation dictionary (0.0 = fully closed,
    1.0 = fully open).  We treat ‘sufficiently open’ as meaning
    that the predicate (handempty) is true.
    """
    obs = task.get_observation()
    open_ratio = obs.gripper_open_amount if hasattr(obs, "gripper_open_amount") else 1.0
    return open_ratio >= empty_threshold


def _release_in_place(env, task):
    """
    Drops the currently held item at the gripper’s present 3-D
    position – a minimal helper that wraps the provided `place`
    skill so we never re-implement motion logic.
    """
    obs = task.get_observation()
    gripper_pos = obs.gripper_pose[:3]
    # `place` is already imported from skill_code; we simply call it
    place(          # noqa: F405  (comes from `from skill_code import *`)
        env,
        task,
        target_pos=gripper_pos,
        approach_distance=0.02,     # very small approach – we are already there
        max_steps=60,
        threshold=0.005,
        approach_axis='-z',
        timeout=5.0,
    )
    # A short pause to ensure physics settles
    for _ in range(5):
        task.step(np.zeros(env.action_shape))


def _ensure_handempty(env, task):
    """
    Makes sure the predicate (handempty) is satisfied before we
    attempt any operation that needs it (e.g. `pick` in the domain
    PDDL).  If the gripper is currently holding something we simply
    open the gripper using the original `place` helper above.
    """
    if _is_hand_empty(task):
        return
    print("[Exploration] Gripper not empty – releasing item first.")
    _release_in_place(env, task)
    print("[Exploration] Release complete, hand should now be empty.")


# ------------------------------------------------------------
# Very small “exploration” routine whose only purpose is to
# discover whether the missing predicate is (handempty).  We
# try to perform a pick without ensuring emptiness; on failure
# we catch the exception, conclude the predicate was absent,
# and re-try after satisfying it.
# ------------------------------------------------------------
def _attempt_pick_with_auto_fix(env, task, obj_name: str, obj_pos: np.ndarray):
    """
    1) Try to pick the object straight away.
    2) If it raises/returns an error (very likely because the
       precondition `(handempty)` was not satisfied), we invoke
       `_ensure_handempty`, then retry exactly once.
    """
    try:
        print(f"[Exploration] First attempt to pick '{obj_name}' …")
        pick(                # noqa: F405  (imported from skill_code)
            env,
            task,
            target_pos=obj_pos,
            approach_distance=0.15,
            max_steps=120,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0,
        )
        print(f"[Exploration] Successfully picked '{obj_name}' on first try.")
    except Exception as exc:
        # Heuristic detection: almost every failure at this point is
        # due to a closed gripper / item already held, i.e. missing
        # `(handempty)` in the state we reasoned with.
        print(f"[Exploration] Pick failed with error: {exc}")
        print("[Exploration] ⇒ Hypothesis: predicate (handempty) missing.")
        # Satisfy predicate, then retry once
        _ensure_handempty(env, task)
        print(f"[Exploration] Retrying pick for '{obj_name}' …")
        pick(                # noqa: F405
            env,
            task,
            target_pos=obj_pos,
            approach_distance=0.15,
            max_steps=120,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0,
        )
        print(f"[Exploration] Successfully picked '{obj_name}' after ensuring hand was empty.")


# ------------------------------------------------------------
# Main entry point
# ------------------------------------------------------------
def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # === Environment Bootstrap ===
    env, task = setup_environment()

    try:
        # Reset the task → get fresh observation
        _, obs = task.reset()

        # Optional: start video capture
        init_video_writers(obs)

        # Wrap `task.step` / `task.get_observation` for video recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # --------------------------------------------------------
        # Retrieve object positions – generic helper provided by
        # the framework.  We do not assume object names ahead of
        # time; we simply iterate over whatever dictionary comes
        # back and showcase the exploration logic.
        # --------------------------------------------------------
        positions = get_object_positions()   # dict: name → np.array(3,)

        # ========================================================
        # EXPLORATION + PLAN EXECUTION
        # ========================================================
        if not positions:
            print("[Warning] No objects returned by get_object_positions(). Nothing to do.")
        else:
            print(f"[Info] Objects discovered: {list(positions.keys())}")

        # Simple demonstration plan:
        #   • For every object we:
        #         – guarantee (handempty)
        #         – pick the object (learning from missing predicate)
        #         – immediately place it back
        #   This is *not* a useful task plan, but it cleanly shows the
        #   discovery & handling of the missing predicate via feedback.
        for obj_name, obj_pos in positions.items():
            print(f"\n=== Handling object: {obj_name} ===")

            # 1) Make sure we start with empty hands
            _ensure_handempty(env, task)

            # 2) Explore pick (auto-fixes missing predicate if necessary)
            _attempt_pick_with_auto_fix(env, task, obj_name, obj_pos)

            # 3) Place it back roughly at its original spot
            print(f"[Task] Placing '{obj_name}' back.")
            place(                # noqa: F405
                env,
                task,
                target_pos=obj_pos,
                approach_distance=0.15,
                max_steps=120,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0,
            )

            # Small idle step between objects
            task.step(np.zeros(env.action_shape))

        print("\n===== Completed demonstration run – all objects visited. =====")

    finally:
        # Always tidy up the simulator no matter what
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
