# run_skeleton_task.py (Completed Version – uses only predefined skills)

import time
import traceback
import numpy as np

from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *        # pick, place, move, rotate, pull (do NOT redefine)
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -----------------------------------------------------------
# Helper utilities – NO new skills are defined below
# -----------------------------------------------------------
def safe_execute(skill_fn, *args, **kwargs):
    """
    Execute any predefined skill while catching and logging exceptions.
    Returns (obs, reward, done).  If an exception occurs, done=True so the
    outer loop terminates gracefully.
    """
    try:
        print(f"[safe_execute] ==> {skill_fn.__name__}")
        return skill_fn(*args, **kwargs)          # call original skill implementation
    except Exception as exc:
        print(f"[safe_execute] EXCEPTION in {skill_fn.__name__}: {exc}")
        traceback.print_exc()
        # Produce dummy values for obs / reward but force ‘done’ so that main loop exits
        return None, 0.0, True


def calibrate_force_if_possible(env):
    """
    Attempt to call an environment-side calibration function if it exists.
    Nothing happens if the env does not expose such method (keeps code generic).
    """
    if hasattr(env, "calibrate_gripper_force"):
        try:
            print("[Calibration] Calibrating gripper force …")
            env.calibrate_gripper_force()
            print("[Calibration] Done.")
        except Exception as exc:
            print(f"[Calibration] Warning – calibration failed: {exc}")


def determine_missing_predicate(observation):
    """
    Very light-weight predicate-detection placeholder.
    We simply inspect known boolean fields on RLBench observation objects
    (if they exist) and infer which of the ‘exploration’ predicates have
    never become TRUE.  The first unseen one is assumed ‘missing’.
    """
    candidate_predicates = [
        "identified_objects",
        "temperature_known_objects",
        "weight_known_objects",
        "durability_known_objects",
        "lock_known_objects",
    ]

    for pred in candidate_predicates:
        if not hasattr(observation, pred):
            return pred          # field completely missing → predicate missing
    # All fields exist ⇒ no obvious missing predicate
    return None


# -----------------------------------------------------------
#                    MAIN TASK ENTRY POINT
# -----------------------------------------------------------
def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # -------------------------------------------------------
    # Environment set-up
    # -------------------------------------------------------
    env, task = setup_environment()
    try:
        # RLBench reset
        descriptions, obs = task.reset()

        # (Optional) video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)               # wrap in recorder
        task.get_observation = recording_get_observation(   # idem for obs
            task.get_observation
        )

        # Perform gripper-force calibration if supported
        calibrate_force_if_possible(env)

        # ---------------------------------------------------
        # Gather initial information
        # ---------------------------------------------------
        positions = get_object_positions()   # {name: (x, y, z)}
        object_names = list(positions.keys())
        print(f"[Init] Objects discovered: {object_names}")

        # Sanity-check from feedback: ensure critical object exists
        must_have_objects = ["item1"]   # feedback explicitly mentioned ‘item1’
        for obj in must_have_objects:
            if obj not in object_names:
                print(f"[Init] Warning – required object '{obj}' missing!")

        # Explore environment to discover missing predicate
        missing_predicate = determine_missing_predicate(obs)
        if missing_predicate:
            print(f"[Exploration] Missing predicate candidate: {missing_predicate}")
        else:
            print("[Exploration] No obvious missing predicate from initial obs.")

        # ---------------------------------------------------
        # Exploration loop
        # ---------------------------------------------------
        robot_waypoint = positions.get("waypoint1", None)
        if robot_waypoint is not None:
            # Simple move so that every object at waypoint becomes ‘identified’
            obs, reward, done = safe_execute(
                move,
                env,
                task,
                target_pos=robot_waypoint,
                approach_distance=0.10,
                max_steps=80,
                threshold=0.01,
                approach_axis="z",
                timeout=5.0,
            )
            if done:
                print("[Exploration] Task finished (done True) while moving.")
                return

        # Iterate over all objects and interact to gain knowledge
        for obj_name, obj_pos in positions.items():
            print(f"\n[Loop] Working on object: {obj_name}")

            # 1) Move close to the object
            obs, reward, done = safe_execute(
                move,
                env,
                task,
                target_pos=obj_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis="z",
                timeout=10.0,
            )
            if done:
                print("[Loop] Leaving – environment reported done after move.")
                return

            # 2) Try to pick it (weight / durability knowledge according to exploration PDDL)
            obs, reward, done = safe_execute(
                pick,
                env,
                task,
                target_pos=obj_pos,
                approach_distance=0.08,
                max_steps=100,
                threshold=0.01,
                approach_axis="z",
                timeout=10.0,
            )
            if done:
                print("[Loop] Leaving – environment reported done after pick.")
                return

            # 3) For drawers or handles, optionally pull
            if "drawer" in obj_name or "handle" in obj_name:
                print(f"[Loop] Attempting pull for {obj_name} (if allowed)")
                obs, reward, done = safe_execute(
                    pull,
                    env,
                    task,
                    target_name=obj_name,
                    max_steps=100,
                    timeout=5.0,
                )
                if done:
                    print("[Loop] Leaving – environment reported done after pull.")
                    return

            # 4) Place the object back roughly at its original position
            obs, reward, done = safe_execute(
                place,
                env,
                task,
                target_pos=obj_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis="z",
                timeout=10.0,
            )
            if done:
                print("[Loop] Leaving – environment reported done after place.")
                return

            # 5) Short pause between objects
            time.sleep(0.2)

        # ---------------------------------------------------
        # Post-exploration – rotate end effector to default pose
        # ---------------------------------------------------
        default_quat = np.array([0.0, 0.0, 0.0, 1.0])   # xyzw (no rotation)
        obs, reward, done = safe_execute(
            rotate,
            env,
            task,
            target_quat=default_quat,
            max_steps=50,
            threshold=0.05,
            timeout=5.0,
        )
        if done:
            print("[Final] Task finished after rotate.")
            return

        print("[Final] Exploration complete – no ‘done’ signal encountered.")

    finally:
        shutdown_environment(env)     # always clean up
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()