# run_skeleton_task.py (Completed Version – includes simple “predicate–exploration” phase)

import re

import time

import numpy as np

# --- DO *NOT* REMOVE / CHANGE THESE IMPORTS -------------------

from pyrep.objects.shape import Shape                       # noqa: F401  (needed by external env)

from pyrep.objects.proximity_sensor import ProximitySensor   # noqa: F401

from env import setup_environment, shutdown_environment

from skill_code import *                                     # noqa: F403

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# --------------------------------------------------------------

# ---------------------------------------------------------------------------

# A.  EXPLORATION-PHASE  –  “find the missing predicate” utility

# ---------------------------------------------------------------------------

EXPLORATION_DOMAIN_PDDL = """

(define (domain exploration)

  (:requirements :strips :typing :conditional-effects :universal-preconditions)

  (:types 

    robot object location

  )

  (:predicates

    (robot-at ?r - robot ?loc - location)

    (at ?obj - object ?loc - location)

    (identified ?obj - object)

    (temperature-known ?obj - object)

    (holding ?obj - object)

    (handempty)

    (weight-known ?obj - object)

    (durability-known ?obj - object)

  )

  (:action move

    :parameters (?r - robot ?from - location ?to - location)

    :precondition (robot-at ?r ?from)

    :effect (and

      (not (robot-at ?r ?from))

      (robot-at ?r ?to)

      (forall (?obj - object)

        (when (at ?obj ?to)

          (identified ?obj)

        )

      )

    )

  )

  (:action move

    :parameters (?r - robot ?from - location ?to - location)

    :precondition (robot-at ?r ?from)

    :effect (and

      (not (robot-at ?r ?from))

      (robot-at ?r ?to)

      (forall (?obj - object)

        (when (at ?obj ?to)

          (temperature-known ?obj)

        )

      )

    )

  )

  (:action pick

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (handempty)

    )

    :effect (and

      (holding ?obj)

      (not (handempty))

      (not (at ?obj ?loc))

      (weight-known ?obj)

    )

  )

  (:action pick

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (handempty)

    )

    :effect (and

      (holding ?obj)

      (not (handempty))

      (not (at ?obj ?loc))

      (durability-known ?obj)

    )

  )

  (:action pull

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (holding ?obj)

       (not (lock-known ?obj))

    )

    :effect (lock-known ?obj)

  )

)

"""

def find_missing_predicates(domain_pddl: str):

    """

    Parse a domain description and return any predicates that appear in

    action preconditions/effects but are *not* declared in the :predicates

    section.

    The parser is deliberately simple – it is sufficient for the narrow

    purpose of this diagnostic step.

    """

    # 1) extract declared predicates

    declared = set()

    predicates_section = re.findall(r'\(:predicates([\s\S]*?)\)', domain_pddl, flags=re.MULTILINE)

    if predicates_section:

        section = predicates_section[0]

        names = re.findall(r'\(([^\s\(\)]+)', section)

        declared.update(names)

    # 2) extract every atom that looks like a predicate symbol

    all_atoms = re.findall(r'\(([^\s\(\)]+)', domain_pddl)

    used = set(all_atoms)

    # 3) remove non-predicate keywords that are caught by the regex

    reserved = {

        ':action', ':parameters', ':precondition', ':effect', 'and',

        'forall', 'when', 'not', 'assign', 'increase', 'decrease'

    }

    used = {p for p in used if p not in reserved}

    # 4) anything in `used` but not in `declared` is missing

    missing = sorted(list(used - declared))

    return missing

# ---------------------------------------------------------------------------

# B.  MAIN EXECUTION  –  (still RLBench-compatible if the environment exists)

# ---------------------------------------------------------------------------

def run_skeleton_task():

    """Generic skeleton for running any task – now extended with an

    exploration phase that detects missing predicates in the given PDDL."""

    print("\n===== Starting Skeleton Task =====\n")

    # ---------------------------------------------------------

    # 1)  Offline exploration / static analysis – find mistakes

    # ---------------------------------------------------------

    print("[Phase] Predicate-Exploration:")

    missing_preds = find_missing_predicates(EXPLORATION_DOMAIN_PDDL)

    if missing_preds:

        print(f"  -> Detected missing predicate(s): {missing_preds}")

    else:

        print("  -> No missing predicates detected!")

    print("---------------------------------------------------------\n")

    # ---------------------------------------------------------

    # 2)  Attempt to bring-up the RLBench environment (optional)

    # ---------------------------------------------------------

    env, task = None, None

    try:

        env, task = setup_environment()           # external helper

        descriptions, obs = task.reset()          # initial reset

        print("[Env] Task descriptions:", descriptions)

        # --- optional video recording setup ------------

        init_video_writers(obs)

        task.step = recording_step(task.step)     # wrapped step

        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------------------------------------

        # 3)  Retrieve relevant object positions

        # ---------------------------------------------

        positions = get_object_positions()

        print("[Env] Retrieved object positions keys:", list(positions.keys()))

        # ---------------------------------------------------------

        # 4)  *Example* skill usage

        #     (Here we just showcase the rotate skill because the

        #      real task specifics are unknown in this skeleton.)

        # ---------------------------------------------------------

        try:

            target_quat = np.array([0.0, 0.0, 0.0, 1.0])  # identity quaternion

            print("[Skill] Executing demo rotate (identity) ...")

            obs, reward, done = rotate(env, task, target_quat,

                                       max_steps=50,

                                       threshold=0.02,

                                       timeout=5.0)       # noqa: F405

            print(f"[Skill] Rotate finished – done={done}, reward={reward}")

        except Exception as e:

            print(f"[Warning] Rotate skill raised an exception: {e}")

        # ---------------------------------------------------------

        # TODO:

        # Insert the real task plan here (move, pick, place, pull, …)

        # ---------------------------------------------------------

    except Exception as e:

        print(f"[Error] Environment set-up or execution failed: {e}")

    finally:

        # Always cleanly shut-down the environment if it was created

        if env is not None:

            shutdown_environment(env)

    print("\n===== End of Skeleton Task =====\n")

# ---------------------------------------------------------------------------

# Script entry point

# ---------------------------------------------------------------------------

if __name__ == "__main__":

    run_skeleton_task()

