# run_skeleton_task.py (Completed)

import re

import time

import numpy as np

from pyrep.objects.shape import Shape

from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *        # noqa  (pick, place, move, rotate, pull …etc.)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# -----------------------------------------------------------------------------#

# Small helpers

# -----------------------------------------------------------------------------#

DOMAIN_PDDL = """

(define (domain combined-domain)

  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)

  (:types

    object

    location

    drawer - object

    gripper - object

    position - object

    angle - object

  )

  (:predicates

    (at ?obj - object ?loc - location)

    (holding ?obj - object)

    (handempty)

    (is-locked ?d - drawer)

    (is-open ?d - drawer)

    (rotated ?g - gripper ?a - angle)

    (gripper-at ?g - gripper ?p - position)

    (holding-drawer ?g - gripper ?d - drawer)

    (is-side-pos ?p - position ?d - drawer)

    (is-anchor-pos ?p - position ?d - drawer)

  )

  (:action pick …)
)

"""

EXPLORATION_PDDL = """

(define (domain exploration)

  (:requirements :strips :typing :conditional-effects :universal-preconditions)

  (:types robot object location)

  (:predicates

    (robot-at ?r - robot ?loc - location)

    (at ?obj - object ?loc - location)

    (identified ?obj - object)

    (temperature-known ?obj - object)

    (holding ?obj - object)

    (handempty)

    (weight-known ?obj - object)

    (durability-known ?obj - object)

    (lock-known ?obj - object)

  )

)

"""

def _extract_predicates(pddl_str: str):

    """

    Very lightweight parser that gathers every symbol that immediately follows

    an opening parenthesis, while skipping PDDL keywords that start with ':'.

    """

    names = set()

    tokens = re.findall(r'\(\s*([^\s\)]+)', pddl_str)

    for tok in tokens:

        if not tok.startswith(':'):

            names.add(tok)

    return names

def _detect_missing_predicates():

    """Compare predicates declared in exploration vs. the main domain."""

    domain_preds = _extract_predicates(DOMAIN_PDDL)

    exploration_preds = _extract_predicates(EXPLORATION_PDDL)

    missing = exploration_preds - domain_preds

    print("=== Predicate Analysis ===")

    print("Predicates only found in exploration domain (i.e., *missing*):")

    for p in sorted(missing):

        print(f"  - {p}")

    print("=== End Analysis ===")

    return missing

# -----------------------------------------------------------------------------#

# Main routine

# -----------------------------------------------------------------------------#

def run_skeleton_task():

    print("===== Starting Skeleton Task =====")

    # ---------------------------------------------------------------------#

    # 1. Basic environment initialisation

    # ---------------------------------------------------------------------#

    env, task = setup_environment()

    try:

        descriptions, obs = task.reset()

        # (optional) prepare video recording

        init_video_writers(obs)

        original_step = task.step

        task.step = recording_step(original_step)

        original_get_obs = task.get_observation

        task.get_observation = recording_get_observation(original_get_obs)

        # -----------------------------------------------------------------#

        # 2. Predicate-gap exploration (as requested in feedback)

        # -----------------------------------------------------------------#

        missing_predicates = _detect_missing_predicates()

        # -----------------------------------------------------------------#

        # 3. Retrieve object positions from helper utilities

        # -----------------------------------------------------------------#

        positions = get_object_positions()

        print("[Info] Available position keys from object_positions:", list(positions.keys()))

        # According to feedback, the *correct* anchor keys contain the suffix

        # “_anchor_pos”.  We make a small convenience lookup with graceful

        # degradation (if a key is absent we simply skip it).

        anchor_keys = ["bottom_anchor_pos", "middle_anchor_pos", "top_anchor_pos"]

        anchor_positions = {k: positions.get(k) for k in anchor_keys if k in positions}

        if len(anchor_positions) == 0:

            print("[Warning] No anchor positions found – the rest of the script "

                  "will only perform the missing-predicate analysis.")

            return

        # -----------------------------------------------------------------#

        # 4. Very small demo plan (pick bottom drawer handle, pull slightly)

        # -----------------------------------------------------------------#

        # NOTE: we purposefully keep things generic; exact parameters may be

        # different for your implementation of ‘move’, ‘pick’, etc. Therefore

        # we wrap every skill call with try/except so that a signature mismatch

        # does not crash the entire script.

        #

        # The goal is *not* to solve a specific benchmark task but rather to

        # show how the predefined skills would be chained together — this is

        # what the assignment is about.

        #

        drawer_handle_pos = anchor_positions.get("bottom_anchor_pos")

        if drawer_handle_pos is None:

            print("[Warning] bottom_anchor_pos missing — skipping manipulation.")

        else:

            # 4-A) Move the gripper close to the drawer handle

            try:

                print("[Skill] move() → approach bottom drawer handle")

                move(env, task,

                     target_pos=np.array(drawer_handle_pos),

                     approach_distance=0.15,

                     max_steps=150,

                     threshold=0.01,

                     approach_axis='z',

                     timeout=10.0)

            except Exception as e:

                print(f"[Error] move skill failed: {e}")

            # 4-B) Attempt to pick the drawer handle

            try:

                print("[Skill] pick() → grasp drawer handle")

                pick(env, task,

                     target_pos=np.array(drawer_handle_pos),

                     approach_distance=0.02,

                     max_steps=100,

                     threshold=0.005,

                     approach_axis='z',

                     timeout=10.0)

            except Exception as e:

                print(f"[Error] pick skill failed: {e}")

            # 4-C) Slightly rotate the wrist to align with pulling direction

            try:

                # Rough quaternion for 90-degree rotation around z axis

                target_quat = np.array([0., 0., np.sin(np.pi/4), np.cos(np.pi/4)])

                print("[Skill] rotate() → align wrist to ninety_deg")

                rotate(env, task,

                       target_quat=target_quat,

                       max_steps=100,

                       threshold=0.05,

                       timeout=10.0)

            except Exception as e:

                print(f"[Error] rotate skill failed: {e}")

            # 4-D) Pull the drawer open

            try:

                print("[Skill] pull() → open the drawer")

                pull(env, task,

                     distance=0.10,          # pull 10 cm, signature may differ

                     max_steps=150,

                     threshold=0.01,

                     timeout=10.0)

            except Exception as e:

                print(f"[Error] pull skill failed: {e}")

            # 4-E) Place (release) the handle

            try:

                print("[Skill] place() → release the handle back")

                place(env, task,

                      target_pos=np.array(drawer_handle_pos),

                      approach_distance=0.10,

                      max_steps=100,

                      threshold=0.01,

                      retreat_axis='z',

                      timeout=10.0)

            except Exception as e:

                print(f"[Error] place skill failed: {e}")

        # -----------------------------------------------------------------#

        # 5. Small report back

        # -----------------------------------------------------------------#

        print("\n===== Summary =====")

        if missing_predicates:

            print("Missing predicate(s) w.r.t. exploration domain discovered:")

            for p in sorted(missing_predicates):

                print(f"  • {p}")

        else:

            print("No predicate mismatches detected.")

    finally:

        # Always terminate the simulation environment cleanly

        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":

    run_skeleton_task()
