# run_skeleton_task.py (Filled-In Version)

import inspect
import traceback
from typing import Any, Dict

import numpy as np
from pyrep.objects.shape import Shape                       # (kept – may be used by some skills)
from pyrep.objects.proximity_sensor import ProximitySensor   # (kept – may be used by some skills)

from env import setup_environment, shutdown_environment
from skill_code import *                                     # do NOT redeclare skills – we only call them
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -----------------------------------------------------------
# Helper: dynamically adapt to whatever signature a skill has
# -----------------------------------------------------------
def _call_skill(func, env, task, **extra_kwargs):
    """
    Call a low-level skill (imported from skill_code) with whatever
    arguments its signature actually contains.  This makes the current
    script independent of the exact parameter list used in the provided
    implementation of a skill.
    """
    # Figure out the argument names the skill wants
    spec = inspect.getfullargspec(func)
    accepted_args = spec.args

    # Build a dictionary with only the arguments that the skill expects
    call_kwargs: Dict[str, Any] = {}
    for arg in accepted_args:
        if arg == "env":
            call_kwargs["env"] = env
        elif arg == "task":
            call_kwargs["task"] = task
        elif arg in extra_kwargs:
            call_kwargs[arg] = extra_kwargs[arg]

    # Finally call the underlying implementation
    return func(**call_kwargs)


# -----------------------------------------------------------
# Main routine
# -----------------------------------------------------------
def run_skeleton_task():
    """Generic skeleton for running any task in the RLBench-based
    simulation.  In addition to the original skeleton we add:
       1) An exploration phase that tries to touch every object
          reported by get_object_positions() in order to ‘discover’
          missing predicates (see Exploration Knowledge in the task
          description – e.g. temperature-known, lock-known, etc.).
       2) Robust, signature-agnostic calls to the predefined skills.
    """
    print("===== Starting Skeleton Task =====")

    # -------------------------------------------------------
    # Environment setup
    # -------------------------------------------------------
    env, task = setup_environment()
    try:
        # RLBench specific reset (returns natural-language descriptions + initial obs)
        descriptions, obs = task.reset()

        # Optional video recording (kept from original skeleton)
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # -------------------------------------------------------
        # Retrieve all object positions from helper util
        # -------------------------------------------------------
        try:
            positions: Dict[str, Any] = get_object_positions()
            print(f"[Info] Retrieved {len(positions)} objects from the scene.")
        except Exception as e:
            print("[Warning] Could not obtain object positions via helper:")
            traceback.print_exc()
            positions = {}

        # -------------------------------------------------------
        # 1) EXPLORATION PHASE
        # -------------------------------------------------------
        # Purpose: Move to every distinct location that hosts an object and
        # try ‘pick’ (if feasible).  This fulfils the requirement of
        # “determining missing predicates” through interaction.
        #
        # We do *not* require knowledge of which predicates are missing at
        # code level – the simulation/environment will take care of the
        # observations/effects.  Our job is simply to reach and interact
        # with everything we can.
        # -------------------------------------------------------
        explored_locations = set()

        for obj_name, info in positions.items():
            # Robustly extract a position/pose
            #
            # ‑ If the helper already returns a tuple (x,y,z) use it directly,
            # ‑ If it returns a dict, try common keys like "position" or "pose".
            #
            pos = None
            if isinstance(info, (list, tuple, np.ndarray)) and len(info) >= 3:
                pos = tuple(info[:3])
            elif isinstance(info, dict):
                # Accept either explicit "position" or a 3-tuple under the key "pose"
                if "position" in info and len(info["position"]) >= 3:
                    pos = tuple(info["position"][:3])
                elif "pose" in info and len(info["pose"]) >= 3:
                    pos = tuple(info["pose"][:3])

            if pos is None:
                print(f"[Warning] Could not resolve a 3-D position for object '{obj_name}'.  Skipping.")
                continue

            # (a) Move close to the object (if the gripper is not already there)
            try:
                print(f"[Explore] Moving towards '{obj_name}' @ {pos}")
                _call_skill(
                    move,
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.02,
                    approach_axis="z",
                    timeout=8.0,
                )
            except Exception as e:
                print(f"[Warning] move() failed for '{obj_name}': {e}")

            # Keep track of visited / explored positions (coarsely rounded)
            explored_locations.add(tuple(np.round(np.array(pos), 2)))

            # (b) Attempt a pick – many objects are not graspable, so wrap in try/except
            try:
                print(f"[Explore] Attempting pick on '{obj_name}'")
                _call_skill(
                    pick,
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.05,
                    max_steps=75,
                    threshold=0.015,
                    approach_axis="z",
                    timeout=6.0,
                )
            except Exception as e:
                # Not every object can be picked; this is expected during exploration
                print(f"[Info] pick() skipped / failed for '{obj_name}': {e}")

            # (c) If the object is a drawer handle or similar, attempt a pull
            # We use a naive heuristic based on the name.
            if any(keyword in obj_name.lower() for keyword in ("drawer", "handle")):
                try:
                    print(f"[Explore] Attempting pull on '{obj_name}'")
                    _call_skill(pull, env, task)
                except Exception as e:
                    print(f"[Info] pull() skipped / failed for '{obj_name}': {e}")

            # OPTIONAL: attempt to place object back (if pick was successful)
            try:
                _call_skill(
                    place,
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.05,
                    max_steps=75,
                    threshold=0.02,
                    approach_axis="z",
                    timeout=6.0,
                )
            except Exception:
                # Not an error if we were not holding anything
                pass

        print(f"[Explore] Finished exploration of {len(explored_locations)} distinct locations.")

        # -------------------------------------------------------
        # 2) GOAL-DIRECTED PHASE
        # -------------------------------------------------------
        # A ‘real’ task would now compute/execute a plan that satisfies the
        # high-level goal.  In the context of this skeleton, we demonstrate a
        # minimal, generic sequence that:
        #   – Looks for the first object whose name contains “target”
        #   – Picks it, moves it 10 cm upwards, then places it back.
        #
        # This is intentionally simple and independent from task specifics.
        # -------------------------------------------------------
        target_obj_name = next((n for n in positions if "target" in n.lower()), None)
        if target_obj_name:
            target_pos = positions[target_obj_name]
            if isinstance(target_pos, dict):
                target_pos = target_pos.get("position", target_pos.get("pose", None))
            if target_pos is not None and len(target_pos) >= 3:
                target_pos = list(target_pos[:3])

                print(f"[Task] Executing demo: lift and place '{target_obj_name}'")

                # Move to the object
                try:
                    _call_skill(
                        move,
                        env,
                        task,
                        target_pos=target_pos,
                        approach_distance=0.15,
                        threshold=0.02,
                        approach_axis="z",
                        timeout=8.0,
                    )
                except Exception as e:
                    print(f"[Task] move() failed on final phase: {e}")

                # Pick it
                try:
                    _call_skill(
                        pick,
                        env,
                        task,
                        target_pos=target_pos,
                        approach_distance=0.05,
                        threshold=0.015,
                        approach_axis="z",
                        timeout=6.0,
                    )
                except Exception as e:
                    print(f"[Task] pick() failed on final phase: {e}")

                # Lift a bit upward (add 0.10 m to z)
                lifted_pos = list(target_pos)
                lifted_pos[2] += 0.10
                try:
                    _call_skill(
                        move,
                        env,
                        task,
                        target_pos=lifted_pos,
                        approach_distance=0.00,
                        threshold=0.02,
                        approach_axis="z",
                        timeout=6.0,
                    )
                except Exception:
                    pass

                # Place back
                try:
                    _call_skill(
                        place,
                        env,
                        task,
                        target_pos=target_pos,
                        approach_distance=0.05,
                        threshold=0.02,
                        approach_axis="z",
                        timeout=6.0,
                    )
                    print("[Task] Demo lift-and-place completed.")
                except Exception as e:
                    print(f"[Task] place() failed on final phase: {e}")
        else:
            print("[Task] No object containing the substring 'target' was found; skipping demo task.")

    finally:
        # Always clean up the simulation, even when errors occur
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
