[Frozen Code Start]

[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys: 'side-pos-bottom','anchor-pos-bottom','table','bin','rubbish'

        # === Compute Quaternion for 90° rotation around Z-axis ===
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # === Execute Oracle Plan ===
        # Step 1: Rotate gripper to 90 degrees
        print("[Step 1] Rotating gripper by 90° around Z-axis")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Step 1] Early termination after rotate.")
            return

        # Step 2: Move gripper to side position of bottom drawer
        side_bottom = np.array(positions['side-pos-bottom'])
        print(f"[Step 2] Moving gripper to side-pos-bottom at {side_bottom}")
        obs, reward, done = move(env, task, side_bottom)
        if done:
            print("[Step 2] Early termination after move.")
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        anchor_bottom = np.array(positions['anchor-pos-bottom'])
        print(f"[Step 3] Moving gripper to anchor-pos-bottom at {anchor_bottom}")
        obs, reward, done = move(env, task, anchor_bottom)
        if done:
            print("[Step 3] Early termination after move.")
            return

        # Step 4: Pull the bottom drawer handle
        print("[Step 4] Pulling bottom drawer handle to open drawer")
        # A small pull distance along the X-axis; adjust distance if needed
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Step 4] Early termination after pull.")
            return

        # Step 5: Move gripper back above the table for picking
        table_pos = np.array(positions['table'])
        print(f"[Step 5] Moving gripper above table at {table_pos}")
        obs, reward, done = move(env, task, table_pos)
        if done:
            print("[Step 5] Early termination after move.")
            return

        # Step 6: Pick up the rubbish from the table
        rubbish_pos = np.array(positions['rubbish'])
        print(f"[Step 6] Picking up rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Step 6] Early termination after pick.")
            return

        # Step 7: Move gripper to the bin location
        bin_pos = np.array(positions['bin'])
        print(f"[Step 7] Moving gripper to bin at {bin_pos}")
        obs, reward, done = move(env, task, bin_pos)
        if done:
            print("[Step 7] Early termination after move.")
            return

        # Step 8: Place the rubbish into the bin
        print(f"[Step 8] Placing rubbish into bin at {bin_pos}")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Step 8] Task finished after place. Reward:", reward)
        else:
            print("[Step 8] Place executed, final done flag:", done, "Reward:", reward)

    except Exception as e:
        print("An exception occurred during task execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()