from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from skill_code import move, pick, place, rotate, pull
import numpy as np

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Map required positions
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: Rotating gripper to 90 degrees")
        # Quaternion for 90° about Z axis: xyzw = [0, 0, sin(π/4), cos(π/4)]
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Plan] Task ended after rotate. Reward:", reward)
            return

        # Step 2: Move to side position of bottom drawer
        print("[Plan] Step 2: Moving to side position of bottom drawer:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Plan] Task ended after move-to-side. Reward:", reward)
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Plan] Step 3: Moving to anchor position of bottom drawer:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick the drawer handle (grasp at anchor)
        print("[Plan] Step 4: Picking drawer handle at anchor position")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after pick-drawer. Reward:", reward)
            return

        # Step 5: Pull to open the drawer
        print("[Plan] Step 5: Pulling the drawer to open")
        # Use a nominal pull distance (e.g., 0.1m) along the x-axis
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Task ended after pull. Reward:", reward)
            return

        # Step 6: Pick up the rubbish on the table
        print("[Plan] Step 6: Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Task ended after pick rubbish. Reward:", reward)
            return

        # Step 7: Place the rubbish into the bin
        print("[Plan] Step 7: Placing rubbish into bin at:", bin_pos)
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Task finished all steps but done=False. Current reward:", reward)

    except Exception as e:
        print("[Error] Exception during plan execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()