[Frozen Code Start]

[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running the drawer-opening and rubbish-disposal task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We assume 'bottom_side_pos', 'bottom_anchor_pos', 'item3'/'rubbish', and 'bin' exist
        bottom_side = np.array(positions['bottom_side_pos'])
        bottom_anchor = np.array(positions['bottom_anchor_pos'])
        bin_pos = np.array(positions['bin'])
        # The rubbish object may be under key 'rubbish' or 'item3'
        rubbish_key = 'rubbish' if 'rubbish' in positions else 'item3'
        rubbish_pos = np.array(positions[rubbish_key])

        # === Plan Execution ===
        # Step 1: Rotate gripper to 90 degrees around the x-axis
        print("[Task] Step 1: rotate to 90°")
        target_quat = R.from_euler('x', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended after rotate.")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Step 2: move to bottom side position", bottom_side)
        obs, reward, done = move(env, task, bottom_side)
        if done:
            print("[Task] Ended after move-to-side.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Step 3: move to bottom anchor position", bottom_anchor)
        obs, reward, done = move(env, task, bottom_anchor)
        if done:
            print("[Task] Ended after move-to-anchor.")
            return

        # Step 4: Pick (grasp) the drawer handle
        print("[Task] Step 4: pick drawer handle at", bottom_anchor)
        obs, reward, done = pick(env, task, bottom_anchor)
        if done:
            print("[Task] Ended after pick-drawer.")
            return

        # Step 5: Pull the drawer open along the x-axis by 0.2 meters
        print("[Task] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Ended after pull.")
            return

        # Step 6: Pick up the rubbish item
        print("[Task] Step 6: pick rubbish at", rubbish_pos)
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Task] Ended after pick rubbish.")
            return

        # Step 7: Move to the bin location
        print("[Task] Step 7: move to bin at", bin_pos)
        obs, reward, done = move(env, task, bin_pos)
        if done:
            print("[Task] Ended after move to bin.")
            return

        # Step 8: Place the rubbish into the bin
        print("[Task] Step 8: place rubbish in bin")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task finished but done=False. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()