from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Identify positions of relevant objects
        # Drawer positions
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        # Rubbish item (assumed to be 'item3')
        rubbish_pos = positions['item3']
        # Bin location
        bin_pos = positions['bin']

        # === Task Plan Execution ===

        # 1) Rotate gripper to ninety degrees to prepare for drawer manipulation
        ninety_shape = Shape('ninety_deg')
        ninety_quat = ninety_shape.get_quaternion()
        print("[Task] Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Task ended during rotation.")
            return

        # 2) Move gripper to the side position of the bottom drawer
        print(f"[Task] Moving to bottom drawer side position: {bottom_side_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended while moving to side position.")
            return

        # 3) Move gripper to the anchor position of the bottom drawer
        print(f"[Task] Moving to bottom drawer anchor position: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended while moving to anchor position.")
            return

        # 4) Pull open the bottom drawer
        pull_distance = 0.1  # meters; adjust based on drawer travel distance
        pull_axis = 'x'      # assume pulling along local x-axis
        print(f"[Task] Pulling drawer open by {pull_distance}m along {pull_axis}-axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Task] Task ended during drawer pull.")
            return

        # 5) Pick up the rubbish item (assumed to be 'item3')
        approach_distance = 0.15
        approach_axis = 'z'
        print(f"[Task] Picking up rubbish at: {rubbish_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=approach_distance,
            approach_axis=approach_axis
        )
        if done:
            print("[Task] Task ended during pick.")
            return

        # 6) Place the rubbish into the bin
        print(f"[Task] Placing rubbish into bin at: {bin_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=approach_distance,
            approach_axis=approach_axis
        )
        if done:
            print("[Task] Task ended during place.")
            return

        # Final check
        if reward is not None and reward >= 0:
            print(f"[Task] Completed plan successfully! Final reward: {reward}")
        else:
            print("[Task] Plan executed but did not receive success signal.")

    except Exception as e:
        print(f"[Task] Encountered exception: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()