[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys: 'bottom_side_pos','bottom_anchor_pos','rubbish','bin'
        
        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to 90 degrees about Z")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Episode ended after rotation.")
            return

        # Step 2: Move to side position of bottom drawer
        bottom_side_pos = positions['bottom_side_pos']
        print(f"[Task] Step 2: Moving gripper to bottom side position {bottom_side_pos}")
        obs, reward, done = move(env, task, np.array(bottom_side_pos))
        if done:
            print("[Task] Episode ended during move-to-side.")
            return

        # Step 3: Move to anchor position of bottom drawer
        bottom_anchor_pos = positions['bottom_anchor_pos']
        print(f"[Task] Step 3: Moving gripper to bottom anchor position {bottom_anchor_pos}")
        obs, reward, done = move(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Task] Episode ended during move-to-anchor.")
            return

        # Step 4: Pick the drawer handle (treat as pick on anchor position)
        print("[Task] Step 4: Picking drawer handle at anchor position")
        obs, reward, done = pick(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Task] Episode ended during pick-drawer.")
            return

        # Step 5: Pull the drawer open by a fixed distance along X
        pull_distance = 0.2  # adjust as needed for full drawer travel
        print(f"[Task] Step 5: Pulling drawer by {pull_distance} along +X axis")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Episode ended during pull.")
            return

        # Step 6: Pick the rubbish from the table
        rubbish_pos = positions['rubbish']
        print(f"[Task] Step 6: Picking up rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, np.array(rubbish_pos))
        if done:
            print("[Task] Episode ended during pick(rubbish).")
            return

        # Step 7: Place the rubbish into the bin
        bin_pos = positions['bin']
        print(f"[Task] Step 7: Placing rubbish into bin at {bin_pos}")
        obs, reward, done = place(env, task, np.array(bin_pos))
        if done:
            print("[Task] Episode ended during place(rubbish).")
            return

        # Final outcome
        if reward is not None:
            print(f"[Task] Task completed. Final reward: {reward}")
        else:
            print("[Task] Task finished without done=True but steps executed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()   