
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and prepare for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to 90 degrees")
        ninety_quat = R.from_euler('y', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Plan] Task ended unexpectedly after rotate.")
            return

        # Step 2: Move to bottom side position
        print("[Plan] Step 2: move to bottom side position")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Task ended unexpectedly after move-to-side.")
            return

        # Step 3: Move to bottom anchor position
        print("[Plan] Step 3: move to bottom anchor position")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Task ended unexpectedly after move-to-anchor.")
            return

        # Step 4: Pick the drawer handle at anchor position
        print("[Plan] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Task ended unexpectedly after pick-drawer.")
            return

        # Step 5: Pull to open the drawer
        print("[Plan] Step 5: pull to open drawer")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Task ended unexpectedly after pull.")
            return

        # Step 6: Pick up the rubbish (item3)
        print("[Plan] Step 6: pick up rubbish (item3)")
        rubbish_pos = np.array(positions['item3'])
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Plan] Task ended unexpectedly after pick rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        print("[Plan] Step 7: place rubbish into bin")
        bin_pos = np.array(positions['bin'])
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Plan] Task ended after place. Reward:", reward)
        else:
            print("[Plan] Finished plan execution. Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
