from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pull, pick, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    print("===== Starting Combined Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # Plan Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan Step 1] rotate gripper")
        # quaternion for +90° about Z: [0,0,sin45,cos45]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Terminated after rotate.")
            return

        # Plan Step 2: Move to bottom side position
        print("[Plan Step 2] move to bottom side position")
        side_bottom_pos = positions['side-pos-bottom']
        obs, reward, done = move(env, task, np.array(side_bottom_pos))
        if done:
            print("[Plan] Terminated after moving to side position.")
            return

        # Plan Step 3: Move to bottom anchor position
        print("[Plan Step 3] move to bottom anchor position")
        anchor_bottom_pos = positions['anchor-pos-bottom']
        obs, reward, done = move(env, task, np.array(anchor_bottom_pos))
        if done:
            print("[Plan] Terminated after moving to anchor position.")
            return

        # Plan Step 4: Pull the bottom drawer open
        print("[Plan Step 4] pull bottom drawer")
        joint_bottom_pos = positions.get('bottom_joint_pos', None)
        if joint_bottom_pos is not None:
            pull_distance = float(np.linalg.norm(np.array(joint_bottom_pos) - np.array(anchor_bottom_pos)))
        else:
            pull_distance = 0.1
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Plan] Terminated after pull.")
            return

        # Plan Step 5: Pick up the rubbish
        print("[Plan Step 5] pick up rubbish")
        rubbish_pos = positions['rubbish']
        obs, reward, done = pick(env, task, np.array(rubbish_pos))
        if done:
            print("[Plan] Terminated after pick.")
            return

        # Plan Step 6: Place the rubbish in the bin
        print("[Plan Step 6] place rubbish in bin")
        bin_pos = positions['bin']
        obs, reward, done = place(env, task, np.array(bin_pos))
        if done:
            print("[Plan] Task completed successfully! Final reward:", reward)
        else:
            print("[Plan] Executed all steps; done flag is", done, "final reward:", reward)

    except Exception as e:
        print(f"[Plan] Exception during execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Combined Task =====")

if __name__ == "__main__":
    run_skeleton_task()