import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print(f"[Task] Rotating gripper to ninety_deg (quat={ninety_quat})")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Episode ended during rotate.")
            return

        # Step 2: Move to drawer side position
        side_key = 'bottom_side_pos'
        if side_key not in positions:
            print(f"[Task] Side position '{side_key}' not found; cannot open drawer.")
            return
        side_pos = np.array(positions[side_key])
        print(f"[Task] Moving to drawer side position '{side_key}' at {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Episode ended during move-to-side.")
            return

        # Step 3: Move to drawer anchor position
        anchor_key = 'bottom_anchor_pos'
        if anchor_key not in positions:
            print(f"[Task] Anchor position '{anchor_key}' not found; cannot open drawer.")
            return
        anchor_pos = np.array(positions[anchor_key])
        print(f"[Task] Moving to drawer anchor position '{anchor_key}' at {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Episode ended during move-to-anchor.")
            return

        # Step 4: Pick the drawer handle (simulate pick-drawer)
        print("[Task] Grasping drawer handle via pick() at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos
        )
        if done:
            print("[Task] Episode ended during pick-drawer.")
            return

        # Step 5: Pull to open the drawer
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.1,
            pull_axis='x'
        )
        if done:
            print("[Task] Episode ended during pull.")
            return
        print("[Task] Drawer opened successfully.")

        # Step 6: Pick up the rubbish object
        rubbish_key = 'item3'
        if rubbish_key not in positions:
            print(f"[Task] Rubbish '{rubbish_key}' not found; aborting.")
            return
        rubbish_pos = np.array(positions[rubbish_key])
        print(f"[Task] Picking up rubbish '{rubbish_key}' at {rubbish_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos
        )
        if done:
            print("[Task] Episode ended during pick of rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        bin_key = 'bin'
        if bin_key not in positions:
            print(f"[Task] Bin '{bin_key}' not found; aborting place.")
            return
        bin_pos = np.array(positions[bin_key])
        print(f"[Task] Placing rubbish into bin at {bin_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos
        )
        if done:
            print("[Task] Episode ended during place.")
            return
        print(f"[Task] Rubbish placed into bin successfully. Final reward: {reward}")
        print("[Task] Goal achieved: drawer opened and rubbish dropped into bin.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
