import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Map logical names to positions
        middle_side_pos   = positions['middle_side_pos']
        middle_anchor_pos = positions['middle_anchor_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # Define quaternions for zero and ninety degree rotations about Z
        zero_quat   = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # === Oracle Plan Execution ===
        # Step 1: rotate gripper from zero to ninety degrees
        print("[Plan] Step 1: rotate gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Episode ended after rotate.")
            return

        # Step 2: move to side position of middle drawer
        print("[Plan] Step 2: move to side position of middle drawer")
        obs, reward, done = move(env, task, middle_side_pos)
        if done:
            print("[Plan] Episode ended after move-to-side.")
            return

        # Step 3: move to anchor position of middle drawer
        print("[Plan] Step 3: move to anchor position of middle drawer")
        obs, reward, done = move(env, task, middle_anchor_pos)
        if done:
            print("[Plan] Episode ended after move-to-anchor.")
            return

        # Step 4: pick the drawer handle (approximate pick-drawer)
        print("[Plan] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, middle_anchor_pos)
        if done:
            print("[Plan] Episode ended after picking drawer handle.")
            return

        # Step 5: pull to open the drawer
        print("[Plan] Step 5: pull to open drawer")
        # assume pulling along +x by 0.1m opens the drawer
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Episode ended after pull.")
            return

        # Step 6: pick the rubbish from the table
        print("[Plan] Step 6: pick the rubbish")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Episode ended after picking rubbish.")
            return

        # Step 7: place the rubbish into the bin
        print("[Plan] Step 7: place the rubbish in the bin")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Plan] Episode ended after placing rubbish.")
            return

        print("[Task] All steps completed. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()