import numpy as np
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and initialize recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions
        positions = get_object_positions()
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper from zero_deg to ninety_deg")
        target_quat = R.from_euler('y', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task ended during rotate.")
            return

        # Step 2: Move to side position of bottom drawer
        print("[Plan] Step 2: move to side position of bottom drawer")
        obs, reward, done = move(env, task, np.array(bottom_side_pos))
        if done:
            print("[Plan] Task ended during move-to-side.")
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Plan] Step 3: move to anchor position of bottom drawer")
        obs, reward, done = move(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Plan] Task ended during move-to-anchor.")
            return

        # Step 4: Pick the drawer handle
        print("[Plan] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Plan] Task ended during pick-drawer.")
            return

        # Step 5: Pull the drawer open
        print("[Plan] Step 5: pull drawer open")
        pull_distance = 0.1
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task ended during pull.")
            return

        # Step 6: Pick up the rubbish from the table
        print("[Plan] Step 6: pick rubbish from table")
        obs, reward, done = pick(env, task, np.array(rubbish_pos))
        if done:
            print("[Plan] Task ended during pick rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        print("[Plan] Step 7: place rubbish into bin")
        obs, reward, done = place(env, task, np.array(bin_pos))
        if done:
            print("[Plan] Task ended during place rubbish.")
            return

        # Final
        print("[Plan] All steps executed. Final reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()