[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset environment and initialize recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()

        # Prepare mapping from named angles to quaternions
        angle_quat_map = {
            'zero_deg': R.from_euler('z', 0, degrees=True).as_quat(),
            'ninety_deg': R.from_euler('z', 90, degrees=True).as_quat()
        }

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to ninety_deg")
        target_quat = angle_quat_map.get('ninety_deg')
        if target_quat is None:
            raise KeyError("Quaternion for 'ninety_deg' not found")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Episode ended after rotate")
            return

        # Step 2: Move to side position of bottom drawer
        print("[Task] Step 2: Moving gripper to bottom side position")
        side_pos = positions.get('bottom_side_pos')
        if side_pos is None:
            raise KeyError("Position for 'bottom_side_pos' not found")
        obs, reward, done = move(env, task, np.array(side_pos))
        if done:
            print("[Task] Episode ended after move-to-side")
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Task] Step 3: Moving gripper to bottom anchor position")
        anchor_pos = positions.get('bottom_anchor_pos')
        if anchor_pos is None:
            raise KeyError("Position for 'bottom_anchor_pos' not found")
        obs, reward, done = move(env, task, np.array(anchor_pos))
        if done:
            print("[Task] Episode ended after move-to-anchor")
            return

        # Step 4: Grasp the bottom drawer handle (pick-drawer)
        print("[Task] Step 4: Grasping drawer handle")
        obs, reward, done = pick(env, task, np.array(anchor_pos))
        if done:
            print("[Task] Episode ended after pick-drawer")
            return

        # Step 5: Pull the drawer open
        print("[Task] Step 5: Pulling the drawer open")
        # Assume drawer opens along +x by 0.1m
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Episode ended after pull")
            return

        # Step 6: Pick up the rubbish on the table
        print("[Task] Step 6: Picking up rubbish on the table")
        rubbish_pos = positions.get('rubbish')
        if rubbish_pos is None:
            raise KeyError("Position for 'rubbish' not found")
        obs, reward, done = pick(env, task, np.array(rubbish_pos))
        if done:
            print("[Task] Episode ended after picking rubbish")
            return

        # Step 7: Place the rubbish into the bin
        print("[Task] Step 7: Placing rubbish into bin")
        bin_pos = positions.get('bin')
        if bin_pos is None:
            raise KeyError("Position for 'bin' not found")
        obs, reward, done = place(env, task, np.array(bin_pos))
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Plan executed but not marked done. Reward:", reward)

    except Exception as e:
        print("[Task] Exception during execution:", e)
        raise
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()