import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Oracle plan execution for pulling open a drawer and disposing rubbish.'''
    print("===== Starting Task: Pull Drawer & Dispose Rubbish =====")
    env, task = setup_environment()
    try:
        # Reset task and initialize video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve 3D positions and orientations
        positions = get_object_positions()
        zero_shape = Shape('zero_deg')
        ninety_shape = Shape('ninety_deg')
        zero_quat = zero_shape.get_quaternion()
        ninety_quat = ninety_shape.get_quaternion()

        # Extract relevant pose targets
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        rubbish_pos = np.array(positions['rubbish'])
        bin_pos = np.array(positions['bin'])

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Step 1] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("Early termination after rotate. Reward:", reward)
            return

        # Step 2: Move gripper to side position of bottom drawer
        print("[Step 2] Moving to side position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("Early termination after move-to-side. Reward:", reward)
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        print("[Step 3] Moving to anchor position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("Early termination after move-to-anchor. Reward:", reward)
            return

        # Step 4: Grip the drawer handle (pick-drawer)
        print("[Step 4] Gripping drawer handle")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("Early termination after pick-drawer. Reward:", reward)
            return

        # Step 5: Pull to open the drawer
        print("[Step 5] Pulling drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("Early termination after pull. Reward:", reward)
            return

        # Step 6: Pick up the rubbish on the table
        print("[Step 6] Picking up rubbish from table")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("Early termination after pick rubbish. Reward:", reward)
            return

        # Step 7: Place the rubbish into the bin
        print("[Step 7] Placing rubbish into bin")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("Task completed successfully! Final reward:", reward)
        else:
            print("Plan executed but task not flagged done. Final reward:", reward)

    except Exception as e:
        print("Exception during task execution:", str(e))
    finally:
        shutdown_environment(env)
        print("===== End of Task =====")

if __name__ == "__main__":
    run_skeleton_task()