import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset task and begin recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve relevant object positions
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # 1) Rotate gripper from zero_deg orientation to 90° around Y axis
        print("[Plan] Rotating gripper to ninety_deg")
        ninety_quat = R.from_euler('y', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Done after rotate. Reward:", reward)
            return

        # 2) Move gripper to the side position of the bottom drawer
        print("[Plan] Moving to side-pos-bottom")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Plan] Done after move-to-side. Reward:", reward)
            return

        # 3) Move gripper to the anchor position of the bottom drawer
        print("[Plan] Moving to anchor-pos-bottom")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Done after move-to-anchor. Reward:", reward)
            return

        # 4) Pick (grasp) the drawer handle at the anchor position
        print("[Plan] Picking drawer handle at anchor-pos-bottom")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Done after pick-drawer. Reward:", reward)
            return

        # 5) Pull to open the drawer fully
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Plan] Done after pull. Reward:", reward)
            return

        # 6) Pick up the rubbish from the table
        print("[Plan] Picking up rubbish")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Plan] Done after pick rubbish. Reward:", reward)
            return

        # 7) Place the rubbish into the bin
        print("[Plan] Placing rubbish into bin")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Finished plan but done=False. Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during plan execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()