import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all object/world positions
        positions = get_object_positions()

        # Precompute quaternions for rotation
        quat_zero = np.array([0.0, 0.0, 0.0, 1.0])
        quat_ninety = np.array([np.sin(np.pi / 4), 0.0, 0.0, np.cos(np.pi / 4)])

        # Step 1: Rotate gripper from zero to ninety degrees
        print("[Step 1] Rotating gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, quat_ninety)
        if done:
            print("Task ended after rotation.")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Step 2] Moving to bottom drawer side position")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("Task ended after moving to side position.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Step 3] Moving to bottom drawer anchor position")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("Task ended after moving to anchor position.")
            return

        # Step 4: Grasp the drawer handle (pick-drawer)
        print("[Step 4] Grasping bottom drawer handle")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("Task ended after grasping drawer handle.")
            return

        # Step 5: Pull the drawer open
        print("[Step 5] Pulling the drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("Task ended after pulling drawer.")
            return

        # Step 6: Pick up the rubbish from the table
        print("[Step 6] Picking up rubbish from table")
        rubbish_pos = np.array(positions['rubbish'])
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("Task ended after picking up rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        print("[Step 7] Placing rubbish into bin")
        bin_pos = np.array(positions['bin'])
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("Task completed successfully! Final reward:", reward)
        else:
            print("Task not completed (done=False). Final reward:", reward)

    except Exception as e:
        print("Error during task execution:", str(e))
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()