import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if desired
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions from the environment
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos       = positions['rubbish']
        bin_pos           = positions['bin']

        # === Plan Execution ===

        # Step 1: rotate gripper from zero_deg to ninety_deg around z-axis
        print("[Task] Step 1: rotate gripper to 90° around Z-axis")
        ninety_z_quat = np.array([0.0, 0.0, 0.70710678, 0.70710678], dtype=np.float32)
        obs, reward, done = rotate(env, task, ninety_z_quat)

        if done:
            print("[Task] Execution ended prematurely after rotate. Reward:", reward)
            return

        # Step 2: move to side position of bottom drawer
        print("[Task] Step 2: move to bottom side position", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Execution ended prematurely after move to side. Reward:", reward)
            return

        # Step 3: move to anchor position of bottom drawer
        print("[Task] Step 3: move to bottom anchor position", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Execution ended prematurely after move to anchor. Reward:", reward)
            return

        # Step 4: pick the drawer handle at the anchor position
        print("[Task] Step 4: pick drawer handle at anchor pos", bottom_anchor_pos)
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Execution ended prematurely after pick handle. Reward:", reward)
            return

        # Step 5: pull the drawer open along +X by 0.05 m
        print("[Task] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.05, pull_axis='x')
        if done:
            print("[Task] Execution ended prematurely after pull. Reward:", reward)
            return

        # Step 6: pick up the rubbish on the table
        print("[Task] Step 6: pick rubbish at", rubbish_pos)
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Task] Execution ended prematurely after pick rubbish. Reward:", reward)
            return

        # Step 7: place the rubbish into the bin
        print("[Task] Step 7: place rubbish into bin at", bin_pos)
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Plan finished but environment not done. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()