import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the combined open-drawer and disposal task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_bottom_pos = positions.get('bottom_side_pos')
        if side_bottom_pos is None:
            raise KeyError("Cannot find side position for bottom drawer.")
        anchor_bottom_pos = positions.get('bottom_anchor_pos')
        if anchor_bottom_pos is None:
            raise KeyError("Cannot find anchor position for bottom drawer.")

        # Determine rubbish position (could be labeled 'rubbish' or 'item3')
        if 'rubbish' in positions:
            rubbish_pos = positions['rubbish']
        elif 'item3' in positions:
            rubbish_pos = positions['item3']
        else:
            raise KeyError("Cannot find rubbish position in object positions.")
        bin_pos = positions.get('bin')
        if bin_pos is None:
            raise KeyError("Cannot find bin position in object positions.")

        # Define quaternion for 90° about z-axis in xyzw format
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # === Oracle Plan Execution ===
        # Step 1-4: approach side, rotate, approach anchor, pull (open drawer)
        # Frozen Code Start
    obs, reward, done = move(
            env, task,
            target_pos=side_bottom_pos
        )
obs, reward, done = rotate(
            env, task,
            target_quat=ninety_quat
        )
obs, reward, done = move(
            env, task,
            target_pos=anchor_bottom_pos
        )
obs, reward, done = pull(
            env, task,
            pull_distance=0.1,
            pull_axis='x'
        )
obs, reward, done = pick(
            env, task,
            target_pos=rubbish_pos
        )
    # Frozen Code End

        if done:
            print("[Plan] Task ended unexpectedly after picking rubbish.")
            return

        print("[Plan] Step 6: Placing rubbish into bin at:", bin_pos)
        obs, reward, done = place(
            env, task,
            target_pos=bin_pos
        )
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Task not fully completed. Current reward:", reward)

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()