import numpy as np
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    """Execute the oracle plan: slide open bottom drawer then dispose rubbish."""
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions of all relevant objects
        positions = get_object_positions()

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to 90° about Z")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # returns [x, y, z, w]
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended after rotate with reward", reward)
            return

        # Step 2: Move gripper to bottom drawer side position
        print("[Plan] Step 2: move to bottom drawer side position")
        side_key = 'bottom_side_pos'
        if side_key not in positions:
            raise KeyError(f"{side_key} not found in object positions")
        side_pos = np.array(positions[side_key])
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Ended after move-to-side with reward", reward)
            return

        # Step 3: Move gripper to bottom drawer anchor position
        print("[Plan] Step 3: move to bottom drawer anchor position")
        anchor_key = 'bottom_anchor_pos'
        if anchor_key not in positions:
            raise KeyError(f"{anchor_key} not found in object positions")
        anchor_pos = np.array(positions[anchor_key])
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Ended after move-to-anchor with reward", reward)
            return

        # Step 4: Pick the drawer handle to grasp it
        print("[Plan] Step 4: pick-drawer (grasp handle of bottom drawer)")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Ended after pick-drawer with reward", reward)
            return

        # Step 5: Pull drawer open along +X by 0.05m
        print("[Plan] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.05, pull_axis='x')
        if done:
            print("[Task] Ended after pull with reward", reward)
            return

        # Step 6: Pick up rubbish from the table
        print("[Plan] Step 6: pick rubbish from table")
        rubbish_key = 'rubbish'
        if rubbish_key in positions:
            rubbish_pos = np.array(positions[rubbish_key])
            obs, reward, done = pick(env, task, rubbish_pos)
        else:
            # Fallback in case 'rubbish' is not reported
            fallback_list = ['item1', 'item2', 'item3']
            picked = False
            for obj in fallback_list:
                if obj in positions:
                    print(f"[Task] Fallback: picking up {obj}")
                    obs, reward, done = pick(env, task, np.array(positions[obj]))
                    picked = True
                    break
            if not picked:
                print("[Task] No rubbish-like object found to pick")
        if done:
            print("[Task] Ended after pick rubbish with reward", reward)
            return

        # Step 7: Place rubbish into the bin
        print("[Plan] Step 7: place rubbish into bin")
        bin_key = 'bin'
        if bin_key not in positions:
            raise KeyError(f"{bin_key} not found in object positions")
        bin_pos = np.array(positions[bin_key])
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task ended without done=True. Reward:", reward)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()