import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Task: Open one drawer and put the rubbish in the bin.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get positions of objects and important waypoints
        positions = get_object_positions()
        # prepare rotation target quaternion (90 degrees about Z)
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # retrieve the positions needed for drawer opening
        if 'bottom_side_pos' not in positions:
            raise KeyError("'bottom_side_pos' not found in positions.")
        bottom_side_pos = positions['bottom_side_pos']
        if 'bottom_anchor_pos' not in positions:
            raise KeyError("'bottom_anchor_pos' not found in positions.")
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # retrieve rubbish and bin positions
        if 'rubbish' not in positions:
            raise KeyError("'rubbish' not found in positions.")
        rubbish_pos = positions['rubbish']
        if 'bin' not in positions:
            raise KeyError("'bin' not found in positions.")
        bin_pos = positions['bin']

        # Step 1: rotate gripper from current orientation to 90° about Z
        print("[Task] Step 1: Rotating gripper to 90° about Z")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Terminated during rotate.")
            return

        # Step 2: move to the side position of the bottom drawer
        print("[Task] Step 2: Moving to side position bottom drawer:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Terminated during move to side pos.")
            return

        # Step 3: move to the anchor position of the bottom drawer
        print("[Task] Step 3: Moving to anchor position bottom drawer:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Terminated during move to anchor pos.")
            return

        # Step 4: grasp and pull the drawer handle
        print("[Task] Step 4: Pulling bottom drawer handle")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Terminated during pull.")
            return

        # Step 5: Pick up the rubbish from the table
        print("[Task] Step 5: Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Task] Terminated during pick.")
            return

        # Step 6: Place the rubbish into the bin
        print("[Task] Step 6: Placing rubbish into bin at:", bin_pos)
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Finished placing but done=False. Reward:", reward)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()