import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()

        # Step 1: Rotate gripper to 90 degrees about Z-axis
        print("[Task] Step 1: rotate gripper to 90 degrees about Z-axis")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        [Frozen Code End]

        if done:
            print("[Task] Terminated after rotate. Reward:", reward)
            return

        # Step 2: Move to bottom side position
        print("[Task] Step 2: move to bottom side position")
        if 'bottom_side_pos' not in positions:
            raise KeyError("bottom_side_pos not found in positions")
        side_pos = np.array(positions['bottom_side_pos'], dtype=np.float32)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Terminated after move-to-side. Reward:", reward)
            return

        # Step 3: Move to bottom anchor position
        print("[Task] Step 3: move to bottom anchor position")
        if 'bottom_anchor_pos' not in positions:
            raise KeyError("bottom_anchor_pos not found in positions")
        anchor_pos = np.array(positions['bottom_anchor_pos'], dtype=np.float32)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Terminated after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick drawer handle (simulate pick-drawer)
        print("[Task] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Terminated after pick-drawer. Reward:", reward)
            return

        # Step 5: Pull drawer open
        print("[Task] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Terminated after pull. Reward:", reward)
            return

        # Step 6: Pick up rubbish on table
        print("[Task] Step 6: pick up rubbish on table")
        if 'rubbish' not in positions:
            raise KeyError("rubbish not found in positions")
        rubbish_pos = np.array(positions['rubbish'], dtype=np.float32)
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Terminated after pick rubbish. Reward:", reward)
            return

        # Step 7: Place rubbish into bin
        print("[Task] Step 7: place rubbish into bin")
        if 'bin' not in positions:
            raise KeyError("bin not found in positions")
        bin_pos = np.array(positions['bin'], dtype=np.float32)
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Finished all steps, done=False. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
        raise
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()