import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate gripper zero_deg -> ninety_deg (about Z axis)
        print("[Task] Rotating gripper to 90 degrees around Z axis")
        half_angle = np.pi / 4
        target_quat = np.array([0.0, 0.0, np.sin(half_angle), np.cos(half_angle)])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Done after rotate. Reward:", reward)
            return

        # Step 2: Move to the side position of bottom drawer
        side_pos = np.array(positions['bottom_side_pos'])
        print(f"[Task] Moving to side position of bottom drawer at {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Done after move-to-side. Reward:", reward)
            return

        # Step 3: Move to the anchor position of bottom drawer
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        print(f"[Task] Moving to anchor position of bottom drawer at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Done after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick the drawer handle (simulate pick-drawer)
        print(f"[Task] Picking drawer handle at {anchor_pos}")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Done after pick-drawer. Reward:", reward)
            return

        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer")
        obs, reward, done = pull(env, task, 0.1)
        if done:
            print("[Task] Done after pull. Reward:", reward)
            return

        # Step 6: Pick up the rubbish from the table
        rubbish_pos = np.array(positions['rubbish'])
        print(f"[Task] Picking up rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Task] Done after pick rubbish. Reward:", reward)
            return

        # Step 7: Place the rubbish into the bin
        bin_pos = np.array(positions['bin'])
        print(f"[Task] Placing rubbish into bin at {bin_pos}")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Completed placing rubbish. Reward:", reward)
        else:
            print("[Task] Task ended without done=True. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()