import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap stepping functions for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Precompute orientation quaternions for rotate step
        angle_quats = {
            'zero_deg': R.from_euler('xyz', [0, 0, 0], degrees=True).as_quat(),
            'ninety_deg': R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()
        }

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        try:
            print("[Task] Step 1: Rotating gripper to ninety_deg orientation")
            obs, reward, done = rotate(env, task, angle_quats['ninety_deg'])
            if done:
                print("[Task] Early termination after rotation. Reward:", reward)
                return
        except Exception as e:
            print("[Error] Rotation failed:", e)
            return

        # Step 2: Move gripper to side position of bottom drawer
        try:
            side_pos = positions['bottom_side_pos']
            print(f"[Task] Step 2: Moving to bottom drawer side position at {side_pos}")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Early termination after move-to-side. Reward:", reward)
                return
        except Exception as e:
            print("[Error] move-to-side failed:", e)
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        try:
            anchor_pos = positions['bottom_anchor_pos']
            print(f"[Task] Step 3: Moving to bottom drawer anchor position at {anchor_pos}")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Early termination after move-to-anchor. Reward:", reward)
                return
        except Exception as e:
            print("[Error] move-to-anchor failed:", e)
            return

        # Step 4: Grasp drawer handle (pick-drawer)
        try:
            print(f"[Task] Step 4: Grasping drawer handle at {anchor_pos}")
            obs, reward, done = pick(env, task, anchor_pos)
            if done:
                print("[Task] Early termination after pick-drawer. Reward:", reward)
                return
        except Exception as e:
            print("[Error] pick-drawer failed:", e)
            return

        # Step 5: Pull the drawer open
        try:
            pull_distance = 0.1
            print(f"[Task] Step 5: Pulling drawer by {pull_distance}m")
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Early termination after pull. Reward:", reward)
                return
        except Exception as e:
            print("[Error] pull failed:", e)
            return

        # Step 6: Pick up the rubbish from table
        try:
            rubbish_pos = positions['rubbish']
            print(f"[Task] Step 6: Picking rubbish at {rubbish_pos}")
            obs, reward, done = pick(env, task, rubbish_pos)
            if done:
                print("[Task] Early termination after picking rubbish. Reward:", reward)
                return
        except Exception as e:
            print("[Error] pick rubbish failed:", e)
            return

        # Step 7: Place the rubbish into the bin
        try:
            bin_pos = positions['bin']
            print(f"[Task] Step 7: Placing rubbish into bin at {bin_pos}")
            obs, reward, done = place(env, task, bin_pos)
            if done:
                print("[Task] Task completed successfully! Final reward:", reward)
            else:
                print("[Task] Completed all steps. Final reward:", reward)
        except Exception as e:
            print("[Error] place rubbish failed:", e)
            return

    finally:
        # Ensure environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()