import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions of all relevant objects
        positions = get_object_positions()

        # define the object to pick and the container
        object_name = 'rubbish'
        container_name = 'bin'

        if object_name not in positions:
            print(f"[Error] Object '{object_name}' not found in positions. Aborting task.")
            return
        if container_name not in positions:
            print(f"[Error] Container '{container_name}' not found in positions. Aborting task.")
            return

        rubbish_pos = positions[object_name]
        bin_pos = positions[container_name]

        # 1) pick up the rubbish
        print(f"[Task] Picking up '{object_name}' at {rubbish_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended during pick. Exiting.")
            return

        # 2) move (if needed) to a safe placement approach pose near the bin
        print(f"[Task] Moving near '{container_name}' at {bin_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.20,
            max_steps=100,
            threshold=0.02,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended during move. Exiting.")
            return

        # 3) place the rubbish into the bin
        print(f"[Task] Placing '{object_name}' into '{container_name}' at {bin_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended during place.")
            return

        print("[Task] Successfully placed rubbish in bin.")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()