import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task in simulation.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset task
        descriptions, obs = task.reset()
        # Initialize video if desired
        init_video_writers(obs)
        # Wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all object positions in the scene
        positions = get_object_positions()
        # Expecting keys for drawer side‐position and handle (anchor) position
        if 'drawer_side_pos' in positions:
            drawer_side_pos = positions['drawer_side_pos']
        elif 'drawer_side' in positions:
            drawer_side_pos = positions['drawer_side']
        else:
            print("[Error] Could not find 'drawer_side_pos' in positions. Available keys:", positions.keys())
            return

        if 'drawer_anchor_pos' in positions:
            drawer_anchor_pos = positions['drawer_anchor_pos']
        elif 'drawer_handle' in positions:
            drawer_anchor_pos = positions['drawer_handle']
        else:
            print("[Error] Could not find 'drawer_anchor_pos' in positions. Available keys:", positions.keys())
            return

        # Define a 90-degree rotation around the gripper’s local Z axis
        # Quaternion format [x, y, z, w]
        ninety_deg_quat = np.array([0.0, 0.0, 0.7071068, 0.7071068])

        # Step 1: Rotate gripper to face the drawer handle
        print("[Task] Rotating gripper to 90 degrees about Z")
        obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.02, timeout=10.0)
        if done:
            print("[Task] Terminated during rotate.")
            return

        # Step 2: Move to the side of the drawer
        print(f"[Task] Moving to drawer side position: {drawer_side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=drawer_side_pos,
            max_steps=150,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move to side.")
            return

        # Step 3: Move to the anchor (handle) position
        print(f"[Task] Moving to drawer handle position: {drawer_anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=drawer_anchor_pos,
            max_steps=150,
            threshold=0.005,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move to anchor.")
            return

        # Step 4: Pick (grasp) the drawer handle
        print(f"[Task] Grasping drawer handle at: {drawer_anchor_pos}")
        obs, reward, done = pick(
            env, task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.003,
            approach_axis='z',
            timeout=8.0
        )
        if done:
            print("[Task] Terminated during pick (grasp).")
            return

        # Step 5: Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(
            env, task,
            target_pos=drawer_anchor_pos,
            pull_distance=0.2,
            max_steps=200,
            threshold=0.01,
            timeout=15.0
        )
        if done:
            print("[Task] Terminated during pull.")
            return

        print("[Task] Drawer opening sequence completed successfully.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()