import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task.'''
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve pose targets and poses from the scene
        positions = get_object_positions()
        # we expect the following keys in positions:
        #   'drawer_side_pos'      : 3‐vector for gripper‐side position
        #   'drawer_anchor_pos'    : 3‐vector for gripper‐anchor position
        #   'gripper_start_quat'   : 4‐vector current gripper orientation
        #   'drawer_target_quat'   : 4‐vector desired orientation (90°)
        #   'drawer_handle_name'   : name or id of the handle object

        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']
        target_quat = positions['drawer_target_quat']
        handle_name = positions['drawer_handle_name']

        # 1) rotate gripper to 90°
        print("[Task] Rotating gripper to target orientation")
        try:
            obs, reward, done = rotate(env,
                                       task,
                                       target_quat,
                                       max_steps=200,
                                       threshold=0.02,
                                       timeout=8.0)
        except Exception as e:
            print(f"[Error] rotate raised exception: {e}")
            return
        if done:
            print("[Task] Early termination during rotate")
            return

        # 2) move to 'side' pre‐grasp position
        print(f"[Task] Moving to side position: {side_pos}")
        try:
            obs, reward, done = move(env,
                                      task,
                                      target_pos=side_pos,
                                      approach_distance=0.10,
                                      max_steps=100,
                                      threshold=0.01,
                                      approach_axis='z',
                                      timeout=5.0)
        except Exception as e:
            print(f"[Error] move to side_pos raised exception: {e}")
            return
        if done:
            print("[Task] Early termination moving to side")
            return

        # 3) move to 'anchor' grasp position
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        try:
            obs, reward, done = move(env,
                                      task,
                                      target_pos=anchor_pos,
                                      approach_distance=0.02,
                                      max_steps=100,
                                      threshold=0.005,
                                      approach_axis='z',
                                      timeout=5.0)
        except Exception as e:
            print(f"[Error] move to anchor_pos raised exception: {e}")
            return
        if done:
            print("[Task] Early termination moving to anchor")
            return

        # 4) pick (grasp) the drawer handle
        print(f"[Task] Grasping drawer handle '{handle_name}'")
        try:
            obs, reward, done = pick(env,
                                      task,
                                      target_pos=anchor_pos,
                                      grip_object=handle_name,
                                      max_steps=80,
                                      threshold=0.005,
                                      approach_axis='y',
                                      timeout=5.0)
        except Exception as e:
            print(f"[Error] pick raised exception: {e}")
            return
        if done:
            print("[Task] Early termination during pick")
            return

        # 5) pull the drawer open
        print("[Task] Pulling drawer open")
        try:
            obs, reward, done = pull(env,
                                      task,
                                      pull_distance=0.15,
                                      max_steps=100,
                                      threshold=0.01,
                                      approach_axis='x',
                                      timeout=7.0)
        except Exception as e:
            print(f"[Error] pull raised exception: {e}")
            return
        if done:
            print("[Task] Early termination during pull")
            return

        print("[Task] Drawer should now be open.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
