import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Retrieved position keys:", list(positions.keys()))

        # Replace these keys with those matching your environment
        # The positions dict should contain:
        #   'drawer_side_pos'    -- a 3D waypoint at the side of the drawer handle
        #   'drawer_anchor_pos'  -- a 3D waypoint at the anchor (handle) of the drawer
        #   'object_in_drawer'   -- the 3D position of the target object inside the drawer
        #   'place_position'     -- the 3D goal location to place the object
        try:
            drawer_side = positions['drawer_side_pos']
            drawer_anchor = positions['drawer_anchor_pos']
            object_in_drawer = positions['object_in_drawer']
            place_pos = positions['place_position']
        except KeyError as e:
            print(f"[Error] Missing expected position key: {e}")
            return

        # === Oracle Plan Execution ===

        # 1) Rotate gripper so it can approach the drawer handle correctly
        #    Assuming a 90-degree rotation around one axis
        ninety_deg_quat = [0.0, 0.0, 0.7071, 0.7071]
        print("[Step 1] Rotating gripper by 90 degrees...")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat,
                                   max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Done] Episode ended during rotate.")
            return

        # 2) Move gripper to a side position near the drawer
        print("[Step 2] Moving to drawer side position:", drawer_side)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side,
                                 max_steps=150,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Done] Episode ended during move to side.")
            return

        # 3) Move gripper to the anchor (handle) of the drawer
        print("[Step 3] Moving to drawer anchor position:", drawer_anchor)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor,
                                 max_steps=150,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Done] Episode ended during move to anchor.")
            return

        # 4) Pick the drawer handle
        print("[Step 4] Picking drawer handle...")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Done] Episode ended during handle pick.")
            return

        # 5) Pull the drawer open
        print("[Step 5] Pulling drawer open...")
        obs, reward, done = pull(env, task)
        if done:
            print("[Done] Episode ended during pull.")
            return

        # 6) Pick the object inside the open drawer
        print("[Step 6] Picking object inside drawer at:", object_in_drawer)
        obs, reward, done = pick(env, task,
                                 target_pos=object_in_drawer,
                                 approach_distance=0.08,
                                 max_steps=150,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=8.0)
        if done:
            print("[Done] Episode ended during inside-object pick.")
            return

        # 7) Place the object at the target place position
        print("[Step 7] Placing object at:", place_pos)
        obs, reward, done = place(env, task,
                                  target_pos=place_pos,
                                  approach_distance=0.10,
                                  max_steps=150,
                                  threshold=0.01,
                                  approach_axis='z',
                                  timeout=8.0)
        if done:
            print("[Done] Episode ended during place.")
            return

        print("===== Task Completed Successfully =====")

    except Exception as e:
        print(f"[Exception] An error occurred during execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()