import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, move_to_side, move_to_anchor, pick_drawer, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("Available positions:", positions.keys())

        # identify the names/keys in the positions dict
        # assume there is exactly one drawer and one gripper in this task
        drawer_name = next(k for k in positions if 'drawer' in k and 'side' not in k and 'anchor' not in k)
        gripper_name = next(k for k in positions if 'gripper' in k)
        side_pos_key = next(k for k in positions if drawer_name in k and 'side' in k)
        anchor_pos_key = next(k for k in positions if drawer_name in k and 'anchor' in k)

        drawer_pos = positions[drawer_name]
        side_position = positions[side_pos_key]
        anchor_position = positions[anchor_pos_key]

        print(f"[Task] Drawer '{drawer_name}' at {drawer_pos}")
        print(f"[Task] Side position: {side_position}, Anchor position: {anchor_position}")

        # === Oracle Plan ===

        # 1) Rotate the gripper to 90 degrees about its wrist.
        # we'll use quaternion [0,0,sin(45°),cos(45°)] = [0,0,0.7071,0.7071]
        target_quat = [0.0, 0.0, 0.7071, 0.7071]
        print("[Plan] Rotating gripper to 90° orientation")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Ended after rotate")
            return

        # 2) Move gripper from its current pose to the side of the drawer
        print("[Plan] Moving to side of drawer")
        obs, reward, done = move_to_side(env, task,
                                         gripper_name,
                                         drawer_name,
                                         drawer_pos,
                                         side_position)
        if done:
            print("[Plan] Ended after move_to_side")
            return

        # 3) Move gripper from side to anchor position on the drawer
        print("[Plan] Moving to anchor position")
        obs, reward, done = move_to_anchor(env, task,
                                           gripper_name,
                                           drawer_name,
                                           side_position,
                                           anchor_position)
        if done:
            print("[Plan] Ended after move_to_anchor")
            return

        # 4) Pick the drawer handle at the anchor position
        print("[Plan] Grasping drawer handle")
        obs, reward, done = pick_drawer(env, task,
                                        gripper_name,
                                        drawer_name,
                                        anchor_position)
        if done:
            print("[Plan] Ended after pick_drawer")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task,
                                 gripper_name,
                                 drawer_name)
        if done:
            print("[Plan] Ended after pull")
            return

        print("[Plan] Drawer should now be open – task complete")

    finally:
        # ensure environment shutdown
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()