# run_skeleton_task.py (plain-text)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We assume get_object_positions() returns a dict of name -> (x,y,z)

        # === Identify relevant positions for the drawer ===
        # According to domain predicates: is-side-pos, is-anchor-pos
        side_positions = [pos for name, pos in positions.items() if 'side' in name]
        anchor_positions = [pos for name, pos in positions.items() if 'anchor' in name]
        if not side_positions or not anchor_positions:
            raise RuntimeError("Failed to find drawer side/anchor positions in object_positions.")

        # Pick the first matching side and anchor positions
        side_pos = side_positions[0]
        anchor_pos = anchor_positions[0]

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper from 'zero_deg' to 'ninety_deg'
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task,
                                   gripper='gripper',        # placeholder name
                                   from_angle='zero_deg',
                                   to_angle='ninety_deg')
        if done:
            print("[Task] Finished after rotate.")
            return

        # 2) Move the gripper to the side position of the drawer
        print(f"[Task] Moving gripper to side position: {side_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=side_pos)
        if done:
            print("[Task] Finished after move to side.")
            return

        # 3) Move the gripper from side to anchor position
        print(f"[Task] Moving gripper to anchor position: {anchor_pos}")
        obs, reward, done = move(env, task,
                                 target_pos=anchor_pos)
        if done:
            print("[Task] Finished after move to anchor.")
            return

        # 4) Pick (grab) the drawer handle at the anchor position
        print("[Task] Picking drawer at anchor position")
        obs, reward, done = pick(env, task,
                                 target_pos=anchor_pos,
                                 approach_distance=0.15,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Finished after pick.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Finished after pull.")
            return

        print("[Task] Drawer open sequence complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()