import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video recording if desired
        init_video_writers(obs)

        # Wrap step and observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions from the scene
        positions = get_object_positions()
        # Expect keys 'drawer_side' and 'drawer_anchor' in this dict
        drawer_side_pos = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']

        # 1) Rotate gripper from zero_deg to ninety_deg to align with drawer handle
        print("[Task] Rotating gripper to 90 degrees")
        # build a quaternion for a rotation of 90° about the gripper's local z-axis
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi / 4), np.cos(np.pi / 4)])
        obs, reward, done = rotate(
            env,
            task,
            target_quat=ninety_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during rotate")
            return

        # 2) Move to the side position of the drawer
        print("[Task] Approaching side of drawer")
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move-to-side")
            return

        # 3) Move to the anchor position (the handle) of the drawer
        print("[Task] Approaching drawer handle (anchor position)")
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move-to-anchor")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        # Define a small linear displacement along the drawer x‑axis for the pull
        pull_end = np.array([
            drawer_anchor_pos[0] + 0.10,
            drawer_anchor_pos[1],
            drawer_anchor_pos[2]
        ])
        obs, reward, done = pull(
            env,
            task,
            target_pos_end=pull_end,
            max_steps=100,
            threshold=0.01,
            approach_axis='x',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during pull")
            return

        print("[Task] Successfully opened the drawer")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()