import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect keys like 'drawer_side', 'drawer_anchor', and maybe robot start location
        side_key = 'drawer_side'
        anchor_key = 'drawer_anchor'
        if side_key not in positions or anchor_key not in positions:
            print(f"[Error] Expected position keys '{side_key}' and '{anchor_key}' in positions dictionary.")
            return

        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # Define quaternions for zero and ninety degrees around the drawer handle axis
        # (these are example values; adjust to your simulation's conventions)
        zero_deg_quat = [0.0, 0.0, 0.0, 1.0]
        ninety_deg_quat = [0.0, 0.7071, 0.0, 0.7071]

        # === Oracle Plan Execution ===

        print("[Plan] Step 1: Ensure gripper is at zero degrees orientation.")
        obs, reward, done = rotate(env, task, zero_deg_quat)
        if done:
            print("[Plan] Early termination during initial rotate.")
            return

        print("[Plan] Step 2: Rotate gripper to ninety_deg for side approach.")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Plan] Early termination during rotate to ninety_deg.")
            return

        # We need a notion of robot's current location. We'll assume obs.robot_pose[:3] is its base.
        try:
            current_base_pos = task.get_observation().robot_pose[:3]
        except Exception:
            # Fallback: use gripper position if robot base not available
            current_base_pos = task.get_observation().gripper_pose[:3]

        print(f"[Plan] Step 3: Move robot from {current_base_pos} to drawer side position {side_pos}.")
        obs, reward, done = move(env, task, tuple(current_base_pos), tuple(side_pos))
        if done:
            print("[Plan] Early termination during move to side.")
            return

        print(f"[Plan] Step 4: Move robot from side position to anchor position {anchor_pos}.")
        obs, reward, done = move(env, task, tuple(side_pos), tuple(anchor_pos))
        if done:
            print("[Plan] Early termination during move to anchor.")
            return

        # Identify the drawer handle object name if provided in descriptions
        drawer_handle = None
        for name in descriptions.get('objects', []):
            if 'drawer_handle' in name or 'drawer' in name:
                drawer_handle = name
                break
        if drawer_handle is None:
            print("[Error] Could not find a drawer handle object in descriptions.")
            return

        print(f"[Plan] Step 5: Pick drawer handle '{drawer_handle}' at anchor position.")
        obs, reward, done = pick(env, task, drawer_handle, tuple(anchor_pos))
        if done:
            print("[Plan] Early termination during pick.")
            return

        print(f"[Plan] Step 6: Pull drawer open.")
        obs, reward, done = pull(env, task, drawer_handle)
        if done:
            print("[Plan] Early termination during pull.")
            return

        print("[Plan] Oracle plan executed successfully.")

    except Exception as e:
        print(f"[Exception] Caught exception during plan execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()