import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import rotate, move_to_side, move_to_anchor, pick_drawer, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and wrap for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # fetch positions of all objects in the scene
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions)

        # identify the drawer and gripper by name in the positions dict
        drawer_name = next((k for k in positions if 'drawer' in k), None)
        gripper_name = next((k for k in positions if 'gripper' in k), None)
        if drawer_name is None or gripper_name is None:
            raise RuntimeError("Could not identify drawer or gripper in positions")

        # identify the side and anchor positions for that drawer
        side_key = next((k for k in positions 
                         if drawer_name in k and 'side' in k), None)
        anchor_key = next((k for k in positions 
                           if drawer_name in k and 'anchor' in k), None)
        if side_key is None or anchor_key is None:
            raise RuntimeError("Could not find side/anchor positions for drawer")

        drawer_side_pos = positions[side_key]
        drawer_anchor_pos = positions[anchor_key]

        # 1) Rotate gripper to align with drawer handle (90 degrees)
        # approximate quaternion for 90° about vertical axis (xyzw)
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Finished early during rotate step.")
            return

        # 2) Move gripper to the side position of the drawer
        obs, reward, done = move_to_side(env, task,
                                         gripper=gripper_name,
                                         drawer=drawer_name,
                                         from_pos=None,     # let the skill infer current
                                         to_pos=drawer_side_pos)
        if done:
            print("[Task] Finished early during move-to-side step.")
            return

        # 3) Move gripper from side to anchor position to grasp handle
        obs, reward, done = move_to_anchor(env, task,
                                           gripper=gripper_name,
                                           drawer=drawer_name,
                                           from_pos=drawer_side_pos,
                                           to_pos=drawer_anchor_pos)
        if done:
            print("[Task] Finished early during move-to-anchor step.")
            return

        # 4) Pick the drawer handle
        obs, reward, done = pick_drawer(env, task,
                                        gripper=gripper_name,
                                        drawer=drawer_name,
                                        position=drawer_anchor_pos)
        if done:
            print("[Task] Finished early during pick-drawer step.")
            return

        # 5) Pull the drawer open
        obs, reward, done = pull(env, task,
                                 gripper=gripper_name,
                                 drawer=drawer_name)
        if done:
            print("[Task] Finished early during pull step.")
            return

        print("===== Task Completed: Drawer is now open =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()