import numpy as np
import math
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve object positions
        positions = get_object_positions()
        if not positions:
            print("[Error] No object positions found.")
            return

        # === Exploration Phase: identify missing predicate ===
        print("[Exploration] Starting exploration to identify missing predicates")
        missing_predicate = None
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Visiting {obj_name} at {obj_pos}")
            try:
                obs, _, done = move(env, task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=50,
                    threshold=0.02,
                    approach_axis='z',
                    timeout=5.0)
                if done:
                    print("[Exploration] Task ended unexpectedly during exploration.")
                    break
                # check for a lock‐known field in observation
                if not hasattr(obs, 'lock_known') and missing_predicate is None:
                    missing_predicate = 'lock-known'
            except Exception as e:
                print(f"[Exploration] Error during move to {obj_name}: {e}")
        print(f"[Exploration] Missing predicate identified: {missing_predicate}")

        # === Oracle Plan Execution ===
        # assume drawer side and anchor positions are labeled in positions dict
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            print("[Error] Drawer positions not found in positions dictionary.")
            return

        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # 1) Rotate gripper to ninety_deg orientation
        # quaternion for 90° about z: [0, 0, sin(45°), cos(45°)]
        quat_ninety = [0.0, 0.0, math.sin(math.pi/4), math.cos(math.pi/4)]
        print("[Task] Rotating gripper to ninety_deg")
        try:
            obs, _, done = rotate(env, task, target_quat=quat_ninety,
                                  max_steps=100, threshold=0.05, timeout=10.0)
            if done:
                print("[Task] Task ended after rotate.")
                return
        except Exception as e:
            print(f"[Task] Error during rotate: {e}")
            return

        # 2) Move to side position of drawer
        print("[Task] Moving to drawer side position")
        try:
            obs, _, done = move(env, task,
                target_pos=drawer_side_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.02,
                approach_axis='z',
                timeout=10.0)
            if done:
                print("[Task] Task ended after move-to-side.")
                return
        except Exception as e:
            print(f"[Task] Error during move-to-side: {e}")
            return

        # 3) Move to anchor position on drawer handle
        print("[Task] Moving to drawer anchor position")
        try:
            obs, _, done = move(env, task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0)
            if done:
                print("[Task] Task ended after move-to-anchor.")
                return
        except Exception as e:
            print(f"[Task] Error during move-to-anchor: {e}")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking the drawer handle")
        try:
            obs, _, done = pick(env, task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=10.0)
            if done:
                print("[Task] Task ended after pick-drawer.")
                return
        except Exception as e:
            print(f"[Task] Error during pick-drawer: {e}")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer")
        try:
            obs, _, done = pull(env, task)
            if done:
                print("[Task] Task ended after pull.")
                return
        except Exception as e:
            print(f"[Task] Error during pull: {e}")
            return

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()