import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset to initial state and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get = task.get_observation
        task.get_observation = recording_get_observation(original_get)

        # Retrieve known positions from the environment
        positions = get_object_positions()
        # These keys must match what object_positions provides
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # 1) Move gripper to the side position of the drawer handle
        print("[Task] Moving gripper to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("[Task] Episode ended during move to side!")
            return

        # 2) Rotate gripper from zero degrees to ninety degrees
        print("[Task] Rotating gripper to ninety degrees")
        # A unit quaternion for a 90° rotation about the appropriate axis
        ninety_deg_quat = [0.0, 0.7071, 0.0, 0.7071]
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Episode ended during rotate!")
            return

        # 3) Move gripper from side position to the anchor position of the drawer handle
        print("[Task] Moving gripper to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Episode ended during move to anchor!")
            return

        # 4) Pull the drawer open
        print("[Task] Pulling the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended during pull!")
            return

        print("[Task] Drawer should now be open.")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()