import numpy as np
import math
from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, place, pull, normalize_quaternion, euler_from_quat
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment and wrap for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        
        # get a dictionary of object positions
        positions = get_object_positions()
        
        # optionally find the gripper and drawer position keys
        gripper_key = next((k for k in positions if "gripper" in k), None)
        side_key     = next((k for k in positions if "side"     in k and "drawer" in k), None)
        anchor_key   = next((k for k in positions if "anchor"   in k and "drawer" in k), None)
        if gripper_key is None or side_key is None or anchor_key is None:
            raise RuntimeError("Could not find gripper or drawer positions in get_object_positions()")
        gripper_home = positions[gripper_key]
        side_pos     = positions[side_key]
        anchor_pos   = positions[anchor_key]
        
        # === Exploration: check which rotated predicate holds ===
        obs = task.get_observation()
        q = normalize_quaternion(obs.gripper_pose[3:7])
        euler = euler_from_quat(q)
        print(f"[Exploration] Current gripper euler angles (rad): {euler}")
        # if yaw (around z) is not ~0, rotate back to zero_deg
        if abs(euler[2]) > 0.05:
            print("[Exploration] Rotating gripper back to zero_deg to satisfy (rotated g zero_deg)")
            zero_quat = normalize_quaternion([0.0, 0.0, 0.0, 1.0])
            obs, reward, done = rotate(env, task, zero_quat)
            if done:
                print("[Task] ended during exploration rotation.")
                return
        
        print("[Exploration] Predicate (rotated g zero_deg) holds; beginning oracle plan.")
        
        # === Oracle plan to open drawer ===
        # 1) rotate to ninety_deg
        ninety_quat = normalize_quaternion([0.0, 0.0, math.sin(math.pi/4), math.cos(math.pi/4)])
        print("[Plan] Rotating to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] ended after rotating to ninety_deg.")
            return
        
        # 2) move gripper from current pose to side position
        print(f"[Plan] Moving gripper to side position: {side_pos}")
        obs, reward, done = move(env, task, gripper_key, gripper_home, side_pos)
        if done:
            print("[Task] ended after move-to-side.")
            return
        
        # 3) move gripper from side position to anchor position
        print(f"[Plan] Moving gripper to anchor position: {anchor_pos}")
        obs, reward, done = move(env, task, gripper_key, side_pos, anchor_pos)
        if done:
            print("[Task] ended after move-to-anchor.")
            return
        
        # 4) pick the drawer handle (anchor)
        print("[Plan] Grasping drawer at anchor position")
        obs, reward, done = pick(env, task, anchor_pos, None, gripper_key)
        if done:
            print("[Task] ended during pick action.")
            return
        
        # 5) pull drawer open
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task, gripper_key, side_key)
        if done:
            print("[Task] ended during pull action.")
            return
        
        print("[Task] Successfully opened the drawer!")
        
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()