import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with the actual keys from your environment
        # For example, positions might contain:
        #   'drawer_side_pos', 'drawer_anchor_pos', 'zero_deg', 'ninety_deg'
        #   'object_1', 'object_2', etc.
        try:
            drawer_side_pos = positions['drawer_side_pos']
            drawer_anchor_pos = positions['drawer_anchor_pos']
            zero_deg = positions['zero_deg']
            ninety_deg = positions['ninety_deg']
            gripper_name = 'gripper'   # name of the gripper object in your sim
            drawer_name = 'drawer'     # name of the drawer object
        except KeyError as e:
            raise RuntimeError(f"Missing expected position key: {e}")

        # === Oracle Plan Execution ===

        # 1) Move gripper to the side position of the drawer
        print("[Plan] Moving to drawer side position")
        obs, reward, done = move(env, task, np.array(drawer_side_pos))
        if done:
            print("[Plan] Task ended prematurely after move to side.")
            return

        # 2) Rotate the gripper from zero_deg to ninety_deg
        print(f"[Plan] Rotating gripper from {zero_deg} to {ninety_deg}")
        obs, reward, done = rotate(env, task, gripper_name, zero_deg, ninety_deg)
        if done:
            print("[Plan] Task ended prematurely after rotate.")
            return

        # 3) Move gripper to the anchor position of the drawer
        print("[Plan] Moving to drawer anchor position")
        obs, reward, done = move(env, task, np.array(drawer_anchor_pos))
        if done:
            print("[Plan] Task ended prematurely after move to anchor.")
            return

        # 4) Pick the drawer handle (anchor) with the gripper
        print("[Plan] Picking drawer handle")
        obs, reward, done = pick(env, task, drawer_anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely after pick-drawer.")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Task ended prematurely after pull.")
            return

        # 6) (Optional) You could now pick/place objects inside the drawer
        #    e.g., object_1_pos = positions['object_1']
        #    obs, reward, done = move(env, task, np.array(object_1_pos))
        #    ...
        #
        # For this skeleton, we assume the goal was just to open the drawer.

        print("===== Plan completed successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()